package dSelf;

import java.util.Vector;
import java.io.*;
import dSelfVM;
 
/**
 * The class ByteVectorSO repesents the byte vector of dSelf.
 * ByteVectorSO is a primitive dSelf object and so it has only one
 * parent slot called "parent", that is shared by all byte vectors.
 * The object, that "parent" refers to has initialy only one
 * parent slot, also called "parent" that refers to the lobby.
 * This object is dedicated to include the user-defined methods,
 * that are shared by all byte vectors.
 */
public class ByteVectorSO extends SerializedPrimitiveSO{

  /** The value of the byte vector object */
  public byte[] byteVector = new byte[0];

 /** The shared parent object for all byte vectors */
  protected static LocalOrdinarySO parent = 
    new LocalOrdinarySO(new ParentSlot("parent", dSelfVM.lobbySO));

 /**
  * The shared slots for all byte vectors. It consists of only one slot
  * called "parent"
  */
  protected static SlotVector slotVector = 
    new SlotVector(new ParentSlot("parent",parent));   
    
 /**
  * The shared parent slots for all byte vectors. It consists of only one slot
  * called "parent"
  */
  protected static Vector parVector = new Vector();   
  
  static{
    parVector.add(parent);   
  }
  
 /**
  * Creates new, empty dSelf byte vector.
  */  
  public ByteVectorSO(){}

 /**
  * Creates new dSelf byte vector with the given Java byte array.
  */  
  public ByteVectorSO(byte[] bytes){
  
    byteVector = bytes;
  }

 /**
  * Returns the slots of the byte vectors. I.e, it's only one slot
  * called "parent".
  */   
  public SlotVector getSlotVector(){
    
    return slotVector; 
  }
  
 /**
  * Returns the shared parent slot for all dSelf byte vectors.
  */  
  protected LocalOrdinarySO getParent(){
  
    return parent;
  }
    
 /**
  * Returns the parent slots of the byte vectors. I.e, it's only one 
  * slot called "parent".
  */   
  public Vector getParentVector(){
  
    return parVector;
  }   

 /**
  * Returns the value of this byte vector in the form "byteVector(...)".
  */
  public String getName(){
  
    String result = "";
    
    for(int i=0; i<byteVector.length; i++)
      result = result + byteVector[i] + (i==byteVector.length-1 ? "" : ",");

    return "byteVector{" + result + "}";
  }

 /**
  * Returns a clone of this byte vector.
  */
  public Object clone(){

    byte[] bV = new byte[byteVector.length];

    for(int i=0; i<byteVector.length; i++)
      bV[i] = byteVector[i];
        
    return new ByteVectorSO(bV);
  }

 /**
  * The dispatcher for dSelf byte vectors, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
      throws dSelfException, NonLocalReturnException{

    try{
     
      switch(msg.getMessageID()){

        case PrimMsg.BYTEAT:                     
	  return _ByteAt(msg.getFirstArg());

        case PrimMsg.BYTEATPUT:                     
	  return _ByteAtPut(msg.getFirstArg(), msg.getSecondArg());

        case PrimMsg.BYTESIZE:                     
	  return _ByteSize();
	
        case PrimMsg.BYTEVECTORASSTRING:                     
	  return _ByteVectorAsString();
	
        case PrimMsg.BYTEVECTORCOMPARE:                     
	  return _ByteVectorCompare(msg.getFirstArg());
	
        case PrimMsg.BYTEVECTORCONCATENATE:                     
	  return _ByteVectorConcatenate(msg.getFirstArg());

        case PrimMsg.CLONEBYTESFILLER:                     
	  return _CloneBytesFiller(msg.getFirstArg(), msg.getSecondArg());

        case PrimMsg.COPYBYTERANGEDSTPOSSRCSRCPOSLENGTH:                     
	  return _CopyByteRangeDstPosSrcSrcPosLength(
	                msg.getFirstArg(), msg.getSecondArg(),
			msg.getThirdArg(), msg.getFourthArg());
	
        case PrimMsg.STRINGPRINT:                     
	  return _StringPrint();
	
      }

      return super.dispatchPrimMsg(msg);       
    }catch(dSelfException e){
      return execFailBlock(msg, e);
    }
  }

 /**
  * Returns the value of the byte vector at the specified position. If 
  * the position is to low or high, a badIndexError will return.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  * @return An IntegerSO, that represents the value of the byte (a
  * number between 0 and 255) at the specified position
  */ 
  protected IntegerSO _ByteAt(dSelfObject arg) throws dSelfException{

    checkIfIntegerSO("_Byte:At:", arg);
    
    try{
      return new IntegerSO((int) byteVector[((IntegerSO)arg).value]);
    }catch(ArrayIndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive \"_ByteAt:\""+
        " is out of bounds !", "Byte:At:", "badIndexError"); 
    }  
  }
 
 /**
  * Puts at the specified position an integer, that must be between
  * 0 and 255. If the position is to low or high, a 'badIndexError' will 
  * return. If the value is to big or small, a 'badTypeError' will be
  * sent to the fail block.
  *  	
  * @param at The argument type must be IntegerSO and is checked by
  * the method. It specifies the index in the vector
  * @param arg An IntegerSO, that represents the value of a byte (a
  * number between 0 and 255)
  * @return This byte vector 
  */
  protected ByteVectorSO _ByteAtPut(dSelfObject at, dSelfObject arg) throws dSelfException{
  
    checkIfIntegerSO("_ByteAt:Put: (1st argument)", at);
    checkIfIntegerSO("_ByteAt:Put: (2nd argument)", arg);

    if(((IntegerSO)arg).value < 0 || ((IntegerSO)arg).value > 255)
      throw new dSelfException("Value of Byte must be between 0 and 255 "+
                 "(_ByteAt: "+((IntegerSO)arg).value+")",
		 "_ByteAt:Put:", "badTypeError"); 

    try{
      byteVector[((IntegerSO)at).value] = (byte) ((IntegerSO)arg).value;
    }catch(ArrayIndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive \"_ByteAt:Put:\""+
         " is out of bounds !", "_ByteAt:Put:", "badIndexError"); 
    }  

    return this;
  }

 /**
  * Returns the size (number of bytes) of this string.
  */
  protected IntegerSO _ByteSize(){
  
    return new IntegerSO(byteVector.length);
  }

 /**
  * Convertes this byte vector to a string. Each byte is treated as
  * an ASCII-character, that will be converted to unicode character.
  */
  protected StringSO _ByteVectorAsString(){

    return new StringSO(new String(byteVector));  
  }
  
 /**
  * Compares this byte vector with the argument, that must also be
  * a byte vector. Returns -1, 0 or 1 if this byte vector is less
  * than, equal to, or greater than the argument, respectively.
  *
  * @param arg The byte vector, that will be compared with this byte vector
  */ 
  protected IntegerSO _ByteVectorCompare(dSelfObject arg) throws dSelfException{
  
    checkIfByteVectorSO("_ByteVectorCompare:", arg);
    
    for(int i=0; 
        i<(byteVector.length < ((ByteVectorSO)arg).byteVector.length ? 
	  byteVector.length : ((ByteVectorSO)arg).byteVector.length);
        i++){
      if(byteVector[i] > ((ByteVectorSO)arg).byteVector[i])
        return new IntegerSO(1);
      
      if(byteVector[i] < ((ByteVectorSO)arg).byteVector[i])    
        return new IntegerSO(-1);       
    }
  
    if(byteVector.length == ((ByteVectorSO)arg).byteVector.length)
      return new IntegerSO(0);
  
    if(byteVector.length < ((ByteVectorSO)arg).byteVector.length)  
      return new IntegerSO(-1);       
    
    return new IntegerSO(1);           
  }    


 /**
  * Concatenates this byte vector and the given argument, which is also a
  * byte vector.
  *
  * @param arg The argument type must be ByteVectorSO and is checked by
  * the method
  * @return The concatenated byte vectors
  */ 	
  protected ByteVectorSO _ByteVectorConcatenate(dSelfObject arg) throws dSelfException{
    
    checkIfByteVectorSO("_ByteVectorConcatenate:", arg);

    byte[] byteVec = new byte[byteVector.length + 
                            ((ByteVectorSO)arg).byteVector.length];
    
    for(int i=0; i <byteVector.length; i++)
      byteVec[i] = byteVector[i];
      
    for(int i=0; i <((ByteVectorSO)arg).byteVector.length; i++)
      byteVec[i+byteVector.length] = ((ByteVectorSO)arg).byteVector[i];
      
    return new ByteVectorSO(byteVec);  
  }

 /**
  * Returns a clone (shallow copy) of this byte vector, possibly
  * resized. The first argument (an IntegerSO) specifies the length
  * of the new byte vector, and the second argument specifies the
  * initial value of extra bytes if the result byte vector is longer
  * than this byte vector. Fails with 'badSizeError' if the first 
  * argument is negative and fails with 'badTypeError' if the second
  * argument isn't between 0 and 255.
  *
  * @param size The size for the new byte vector. The argument 
  * type must be IntegerSO and is checked by the method
  * @param filler The possibly needed filler. The argument type must be 
  * an IntegerSO between 2 and 255 and is checked by the method
  * @return Returns this byte vector
  */  				 
  protected ByteVectorSO _CloneBytesFiller(dSelfObject size, dSelfObject filler) throws dSelfException{
  
    checkIfIntegerSO("_CloneBytes:Filler: (1st argument)", size);
    checkIfIntegerSO("_CloneBytes:Filler: (2nd argument)", filler);
    
    if(((IntegerSO)size).value < 0)
      throw new dSelfException("Value for size must be greater than 0 !",
          "_CloneBytes:Filler:", "badSizeError");
      
    if(((IntegerSO)filler).value < 0 || ((IntegerSO)filler).value > 255)
      throw new dSelfException("Value of byte must be between 0 and 255"+
        " (_CloneBytes:Filler: "+((IntegerSO)filler).value+")", 
        "_CloneBytes:Filler:", "badTypeError");

    byte[] bVec =  new byte[((IntegerSO)size).value];
    
    for(int i=0; i<((IntegerSO)size).value; i++)
      bVec[i] = (byte) (i<byteVector.length ? 
                 byteVector[i] : ((IntegerSO)filler).value);
      
    return new ByteVectorSO(bVec);  
  }
    
 /**
  * Copies the number of bytes, specified by "length", into this byte
  * vector at the position "dstPos" from the "srcVec" byte vector at 
  * position "srcPos". May fail with "badIndexError".
  *
  * @param dstPos The argument type must be IntegerSO and is checked by
  * the method. It specifies the destination index in the vector
  * @param srcVec The argument type must be ByteVectorSO and is checked
  * by the method
  * @param srcPos The argument type must be IntegerSO and is checked by
  * the method. It specifies the source index in the vector
  * @param length The argument type must be IntegerSO and is checked by
  * the method. 
  * @return Returns this byte vector
  */   
  protected ByteVectorSO _CopyByteRangeDstPosSrcSrcPosLength(
        dSelfObject dstPos, dSelfObject srcVec,
	dSelfObject srcPos, dSelfObject length) throws dSelfException{

    checkIfIntegerSO("_CopyByteRangeDstPos:Src:SrcPos:Length: (1st argument)", dstPos);
    checkIfByteVectorSO("_CopyByteRangeDstPos:Src:SrcPos:Length: (2nd argument)", srcVec);
    checkIfIntegerSO("_CopyByteRangeDstPos:Src:SrcPos:Length: (3rd argument)", srcPos);
    checkIfIntegerSO("_CopyByteRangeDstPos:Src:SrcPos:Length: (4th argument)", length);

    try{ 
      for(int i=0 ; i<((IntegerSO)length).value; i++)
        byteVector[((IntegerSO)dstPos).value+i] = 
	  ((ByteVectorSO)srcVec).byteVector[((IntegerSO)srcPos).value+i];
    }catch(ArrayIndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive \""+
        "_CopyByteRangeDstPos:Src:SrcPos:Length:\" is out of bounds !",
	"_CopyByteRangeDstPos:Src:SrcPos:Length", "badIndexError"); 
    }
    
    return this; 		 
  }		 
  
 /**
  * Prints this byte vector at the front-end, that was chosen by the user.
  *
  * @return This byte vector
  */
  protected ByteVectorSO _StringPrint(){
  
    dSelfVM.printMessage(new String(byteVector));
    return this;
  }
}
