package dSelf;

import java.util.Date;
import java.util.Vector;
import dSelfVM;

/**
 * DataSO declares the interfaces and implements the methods,
 * that are shared by primitive and ordinary data objects.
 */
public abstract class DataSO extends DataOrMethodSO implements Cloneable{

  /** A vector for the objects that wait for a lock*/
  private Vector lockedThreads = new Vector();  
  
 /**
  * Locks this object. The argument shall be the owner of the lock
  *
  * @param so The owner of the lock
  */
  protected synchronized void lock(dSelfObject so){
  
    // Get monitor for "so"
    synchronized(so){
      lockedThreads.add(so); 
      
      // If there is another thread running, "so" must wait
      if(lockedThreads.size() > 1){ 
        try{
          so.wait();
        }catch(java.lang.InterruptedException e){/* toDo*/} 
      }	
    }  
  }
  
 /**
  * Unlocks this object. The argument is the owner of the lock
  *
  * @param so The owner of the lock
  */
  protected void unlock(dSelfObject so) throws dSelfException{
  
    // The work is done, so remove "so" if it is the owner of the lock
    if(lockedThreads.remove(0) != so)
      throw new dSelfException("Wrong object tried to remove a lock !",
	    "_Unlock", "primitiveFailedError");

    // Wake up the next object that is waiting
    if(!lockedThreads.isEmpty()){
      synchronized(lockedThreads.firstElement()){
        lockedThreads.firstElement().notify();
      }
    }     
  }
  
 /**
  * Returns the annotation for this object.
  */   
  protected abstract String getAnnotation();   

 /**
  * Evaluates the given dSelf object with the given arguments, that 
  * was found found in a slot with the given name and evaluates it
  * in the context of this object. 
  *
  * @param dSO The object, that shall be evaluated
  * @param args The arguments of the given object. 
  * @return The result of the evaluation of the given object
  */ 
  protected DataSO evalSO(dSelfObject dSO, Vector args, String messagename) 
            throws dSelfException, NonLocalReturnException{
    MethodActivationSO activation = null;
    
    try{
      if(dSO instanceof DataSO)  
        return (DataSO)dSO;

      if(dSO instanceof MethodSO){
        //return ((MethodSO)dSO).getActivation(args, messagename).execute(this);
	activation = ((MethodSO)dSO).getActivation(args, messagename);
	return activation.execute(this);
      }

      if (dSO instanceof AssignmentSO) {
	((AssignmentSO)dSO).setContent((DataSO)args.firstElement());
	return this;
      }

      throw new dSelfException("Internal error: " + dSO
			   + " is neither DataSO, nor MethodSO, nor AssignmentSO");
    }catch(NonLocalReturnException e){
      //System.err.println(this+" "+activation+ "(in DataSO) catch NonLRE");
      if (activation == e.getReturnTarget())
	return e.getReturnValue();
      else
	throw e;
    }
  } 

 /**
  * Check if the given object is an dSelf integer. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfIntegerSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof IntegerSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! An integer was expected.", msg, "badTypeError");
  }			      
    
 /**
  * Check if the given object is an dSelf short. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfShortSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof ShortSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! A short was expected.", msg, "badTypeError");
  }			      
  
 /**
  * Check if the given object is an dSelf long. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfLongSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof LongSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! A long was expected.", msg, "badTypeError");
  }			      
  
 /**
  * Check if the given object is an dSelf float. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfFloatSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof FloatSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! A float was expected.", msg, "badTypeError");
  }			      

 /**
  * Check if the given object is an dSelf double. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfDoubleSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof DoubleSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! A double was expected.", msg, "badTypeError");
  }			      

 /**
  * Check if the given object is an dSelf string. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfStringSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof StringSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! A string was expected.", msg, "badTypeError");
  }			      

 /**
  * Check if the given object is an dSelf block. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfBlockSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof BlockSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! A block was expected.", msg, "badTypeError");
  }			      

 /**
  * Check if the given object is an dSelf byte vector. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfByteVectorSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof ByteVectorSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! A byte vector was expected.", msg, "badTypeError");
  }			      

 /**
  * Check if the given object is an dSelf object vector. If not, then an 
  * error message is created with details about the occurrence of 
  * this error.
  *
  * @param msg The name of primitive message, where the type error  
  * happend.
  * @param dSO The object, whose type is checked
  */  
  protected static void checkIfObjectVectorSO(String msg, dSelfObject dSO) 
            throws dSelfException{
    
    if(!(dSO instanceof ObjectVectorSO))
      throw new dSelfException("Wrong argument-type send to \""+ msg
              + "\"! A object vector was expected.", msg, "badTypeError");
  }			      

 /**
  * Executes the fail block of a primitive message when an error occured.
  * If the message has no fail block, then the given exception is thrown,
  * otherwise the block is executed with two arguments. The first one
  * describes the error and the second one the name of the error.
  *
  * @param msg The message, that caused the error
  * @param e The exception that is thrown, when no fail block was found.
  * @return The return-value of the block after execution
  */
  protected DataSO execFailBlock(PrimMsg msg, dSelfException e) 
            throws dSelfException, NonLocalReturnException{
  
    if(!msg.hasFailBlock())
      throw e;

    Vector vec = new Vector(2);
    vec.add(new StringSO(e.error));
    vec.add(new StringSO(e.name));
    
    checkIfBlockSO("IfFail:", msg.getLastArg());
       	
    return ((BlockSO)msg.getLastArg()).dispatchMsg(
                         new OrdinaryMsg("value:With:", vec));
  }
  
 /**
  * The message dispatcher for primitive messages. It checks, if the
  * message is understood by this object. If yes, then the actions
  * of this are done, otherwise an error message is thrown.
  *
  * @param msg The primitive message, that is sent to this object
  * @return The result of the action, that was caused by the message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
         throws dSelfException, NonLocalReturnException{

    try{
     
      switch(msg.getMessageID()){

        case PrimMsg.SELF:                
	  return this;

        case PrimMsg.CLONE:
	  return _Clone();
	  
        case PrimMsg.CREDITS:                
          _Credits(); 
	  return new NilSO();

        case PrimMsg.CURRENTTIMESTRING:                
	  return _CurrentTimeString(); 

        case PrimMsg.DEBUGCUPON:               
          Globals.debug_cup = true; 
          dSelfVM.printMessage("CUP-debugger is activated");
	  return new NilSO();

        case PrimMsg.DEBUGCUPOFF:               
          Globals.debug_cup = false; 
          dSelfVM.printMessage("CUP-debugger is deactivated");
	  return new NilSO();

        case PrimMsg.DEBUGSEARCHPATHON:           
          Globals.debug_searchPath = true; 
          dSelfVM.printMessage("Search path debugger is activated");
	  return new NilSO();

        case PrimMsg.DEBUGSEARCHPATHOFF:           
          Globals.debug_searchPath = false; 
          dSelfVM.printMessage("Search path debugger is deactivated");
	  return new NilSO();

        case PrimMsg.DEBUGLOOKUPCACHEON:
	  Globals.debug_lookupCache = true; 
	  dSelfVM.printMessage("Lookup cache debugger is activated");
	  return new NilSO();

        case PrimMsg.DEBUGLOOKUPCACHEOFF:           
	  Globals.debug_lookupCache = false; 
	  dSelfVM.printMessage("Lookup cache debugger is deactivated");
	  return new NilSO();
	
        case PrimMsg.DEBUGSCANNERON:           
          Globals.debug_scanner = true; 
          dSelfVM.printMessage("Scanner-debugger is activated");
	  return new NilSO();

        case PrimMsg.DEBUGSCANNEROFF:           
          Globals.debug_scanner = false; 
          dSelfVM.printMessage("Scanner-debugger is deactivated");
	  return new NilSO();

        case PrimMsg.DEBUGFLATPARSETREEON:     
          Globals.debug = true;
          Globals.debug_parsetree_flat = true; 
          dSelfVM.printMessage("Flat parse trees will be generated");
	  return new NilSO();

        case PrimMsg.DEBUGFLATPARSETREEOFF:     
          Globals.debug_parsetree_flat = false; 
          dSelfVM.printMessage("No more flat parse trees will be generated");
	  return new NilSO();

        case PrimMsg.DEBUGINDENTEDPARSETREEOFF: 
          Globals.debug_parsetree_indent = false; 
          dSelfVM.printMessage("No more indented parse trees will be generated");
	  return new NilSO();

        case PrimMsg.DEBUGINDENTEDPARSETREEON: 
          Globals.debug = true;
          Globals.debug_parsetree_indent = true; 
          dSelfVM.printMessage("Indented parse trees will be generated");
	  return new NilSO();

        case PrimMsg.DESCRIBE:                     
	  return _Describe();  

        case PrimMsg.DIRPATH1:                     
	  return _DirPath1();  

        case PrimMsg.DIRPATH2:                     
	  return _DirPath2(msg.getFirstArg());  

        case PrimMsg.EQ:                     
	  return _Eq(msg.getFirstArg());  

        case PrimMsg.FREEMEMORY:                     
	  return _FreeMemory();  

        case PrimMsg.GARBAGECOLLECT:                     
	  return _GarbageCollect();  

        case PrimMsg.GENES:                     
	  return _Genes();

        case PrimMsg.GETLOCATION:                     
	  return _GetLocation();  

        case PrimMsg.GETSLOTNAMES:                     
	  return _GetSlotNames();  

        case PrimMsg.OBJECTID:                     
	  return _ObjectID();  

        case PrimMsg.MEMORY:                     
	  return _Memory();  

        case PrimMsg.OPERATINGSYSTEM:                     
	  return _OperatingSystem();  

        case PrimMsg.PERFORM:
	  return _Perform(msg.getFirstArg());
      
        case PrimMsg.PERFORMWITH1:
        case PrimMsg.PERFORMWITH2:
        case PrimMsg.PERFORMWITH3:
        case PrimMsg.PERFORMWITH4:
        case PrimMsg.PERFORMWITH5:
        case PrimMsg.PERFORMWITH6:
        case PrimMsg.PERFORMWITH7:
        case PrimMsg.PERFORMWITH8:
        case PrimMsg.PERFORMWITH9:
        case PrimMsg.PERFORMWITH10:
        case PrimMsg.PERFORMWITH11:
        case PrimMsg.PERFORMWITH12:
        case PrimMsg.PERFORMWITH13:
        case PrimMsg.PERFORMWITH14:
        case PrimMsg.PERFORMWITH15:
        case PrimMsg.PERFORMWITH16:
        case PrimMsg.PERFORMWITH17:
        case PrimMsg.PERFORMWITH18:
        case PrimMsg.PERFORMWITH19:
        case PrimMsg.PERFORMWITH20:
	  return _PerformWith(msg.getArgs());

        case PrimMsg.PERFORMRESEND:
	  return _PerformResend((StringSO) msg.getFirstArg());
      
        case PrimMsg.PERFORMRESENDWITH1:
        case PrimMsg.PERFORMRESENDWITH2:
        case PrimMsg.PERFORMRESENDWITH3:
        case PrimMsg.PERFORMRESENDWITH4:
        case PrimMsg.PERFORMRESENDWITH5:
        case PrimMsg.PERFORMRESENDWITH6:
        case PrimMsg.PERFORMRESENDWITH7:
        case PrimMsg.PERFORMRESENDWITH8:
        case PrimMsg.PERFORMRESENDWITH9:
        case PrimMsg.PERFORMRESENDWITH10:
        case PrimMsg.PERFORMRESENDWITH11:
        case PrimMsg.PERFORMRESENDWITH12:
        case PrimMsg.PERFORMRESENDWITH13:
        case PrimMsg.PERFORMRESENDWITH14:
        case PrimMsg.PERFORMRESENDWITH15:
        case PrimMsg.PERFORMRESENDWITH16:
        case PrimMsg.PERFORMRESENDWITH17:
        case PrimMsg.PERFORMRESENDWITH18:
        case PrimMsg.PERFORMRESENDWITH19:
        case PrimMsg.PERFORMRESENDWITH20:
	  return _PerformResendWith(msg.getArgs());
      
        case PrimMsg.PERFORMDELEGATINGTO:
	  return _PerformDelegatingTo(msg.getFirstArg(), msg.getSecondArg());
      
        case PrimMsg.PERFORMDELEGATINGTOWITH1:
        case PrimMsg.PERFORMDELEGATINGTOWITH2:
        case PrimMsg.PERFORMDELEGATINGTOWITH3:
        case PrimMsg.PERFORMDELEGATINGTOWITH4:
        case PrimMsg.PERFORMDELEGATINGTOWITH5:
        case PrimMsg.PERFORMDELEGATINGTOWITH6:
        case PrimMsg.PERFORMDELEGATINGTOWITH7:
        case PrimMsg.PERFORMDELEGATINGTOWITH8:
        case PrimMsg.PERFORMDELEGATINGTOWITH9:
        case PrimMsg.PERFORMDELEGATINGTOWITH10:
        case PrimMsg.PERFORMDELEGATINGTOWITH11:
        case PrimMsg.PERFORMDELEGATINGTOWITH12:
        case PrimMsg.PERFORMDELEGATINGTOWITH13:
        case PrimMsg.PERFORMDELEGATINGTOWITH14:
        case PrimMsg.PERFORMDELEGATINGTOWITH15:
        case PrimMsg.PERFORMDELEGATINGTOWITH16:
        case PrimMsg.PERFORMDELEGATINGTOWITH17:
        case PrimMsg.PERFORMDELEGATINGTOWITH18:
        case PrimMsg.PERFORMDELEGATINGTOWITH19:
        case PrimMsg.PERFORMDELEGATINGTOWITH20:
	  return _PerformDelegatingToWith(msg.getArgs());

        case PrimMsg.PRINT:           
          return _Print();     
	
        case PrimMsg.QUIT:                     
	  _Quit();  

        case PrimMsg.TIMEREAL:           
          return _TimeReal();     

        case PrimMsg.LOCK:           
          return _Lock(msg.getFirstArg());     

        case PrimMsg.UNLOCK:           
          return _Unlock(msg.getFirstArg());     
      }
    
      throw new dSelfException("Primitive message \""+msg.getSelector()+
                   "\" was not understood by object <"+getName()+"> !",
		   msg.getSelector(), "badTypeError");   
    }
    catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Locks this object. The argument shall be the owner of the lock
  *
  * @param so The owner of the lock
  */
  protected DataSO _Lock(dSelfObject so){
  
    lock(so);
    return this;
  }
  
 /**
  * Unlocks this object. The argument is the owner of the lock
  *
  * @param so The owner of the lock
  */
  protected DataSO _Unlock(dSelfObject so) throws dSelfException{

    unlock(so);
    return this;
  }
  
 /**
  * Returns a clone of this object.
  */
  protected DataSO _Clone(){
  
    return (DataSO) clone();
  }  

 /**
  * Prints some credits.
  *
  * @return This object
  */
  protected DataSO _Credits(){
  
    dSelfVM.printMessage(dSelfVM.version + "\n" 
      +  "This program is free software; you can redistribute it \n"
      + "and/or modify it under the terms of the GNU General Public \n"
      + "License as published by the Free Software Foundation; \n"
      + "either version 2 of the License, or (at your option) any \n"
      + "later version. \n"
      + "This program is distributed in the hope that it will be \n"
      + "useful, but WITHOUT ANY WARRANTY; without even the implied \n"
      + "warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR \n"
      + "PURPOSE.  See the GNU General Public License for more \n"
      + "details. \n"
      + "You should have received a copy of the GNU General Public \n"
      + "License along with this program; if not, write to the \n"
      + "Free Software Foundation, Inc., 59 Temple Place, \n"
      + "Suite 330, Boston, MA 02111-1307, USA"    
    );
    return this;
  }  

 /**
  * Returns the current time as a string.
  */
  protected StringSO _CurrentTimeString(){
  
    return new StringSO((new Date()).toString());
  }  

 /**
  * Describes all slots of this object by printing all annotations
  * of all slots.
  *
  * @return The dSelf nil object
  */
  protected DataSO _Describe() throws dSelfException{
  
    dSelfVM.printMessage("\nName: " + getAnnotation() + "\n\n"+
        getSlotVector().describeSlots());
    
    return new NilSO();
  }  

 /**
  * Returns a dSelf string with the actual value of the directory path.
  */
  protected StringSO _DirPath1() throws dSelfException{
  
    return new StringSO(dSelfVM.getDirPath());
  }  

 /**
  * Sets a new value for the directory path.
  *
  * @param so The argument type must be StringSO and is checked by
  * the method
  * @return The new directory path
  */
  protected StringSO _DirPath2(dSelfObject so) throws dSelfException{
  
    checkIfStringSO("_DirPath:", so);
    return new StringSO(dSelfVM.setDirPath(((StringSO)so).string));
  }  

 /**
  * Checks if the given object is the same object like this object.
  *
  * @return TrueSO if ot is the same, FalseSO if not.
  */
  protected BooleanSO _Eq(dSelfObject so) throws dSelfException{
  
    // If two objects are located on different VMs, they can't be
    // equal.
    if(!getLocation().equals(so.getLocation()))
      return new FalseSO();
      
    return BooleanSO.asBooleanSO(so.getHashCode() == getHashCode());
  }  
  
 /**
  * Returns the size of free memory of this VM.
  */
  protected LongSO _FreeMemory(){
  
    return new LongSO(Runtime.getRuntime().freeMemory());
  }  

 /**
  * Starts the garbage collector.
  *
  * @return The dSelf nil object
  */
  protected DataSO _GarbageCollect(){
  
    System.gc();
    return new NilSO();
  }  

 /**
  * Returns the genes of this object.
  */
  protected ObjectVectorSO _Genes(){
  
    Long[] tmp = (Long[]) getSlotVector().getGenes().toArray(new Long[0]);
    DataSO[] d = new DataSO[tmp.length];

    for (int i=0; i<tmp.length; i++)
      d[i] = new LongSO(tmp[i].longValue());

    return new LocalObjectVectorSO(d);
  }  

 /**
  * Returns the name of the local VM
  */
  protected StringSO _GetLocation(){
  
    return new StringSO(getLocation());
  }  

 /**
  * Returns an object vector with all names (as StringSO) of this object.
  */
  protected ObjectVectorSO _GetSlotNames(){
  
    return new LocalObjectVectorSO(getSlotVector().getSlotNames());
  }  

 /**
  * Returns the hash code of this object.
  */
  protected IntegerSO _ObjectID(){
  
    return new IntegerSO(hashCode());
  }  

 /**
  * Returns the size of memory of this VM.
  */
  protected LongSO _Memory(){
  
    return new LongSO(Runtime.getRuntime().totalMemory());
  }  

 /**
  * Returns a string with informations about the operating system. If
  * this isn't allowed by the security manager an "securityError" if
  * thrown, that can be caught by an "IfFail:"-block.
  */
  protected StringSO _OperatingSystem() throws dSelfException{
  
    try{
      return new StringSO("OS: "+System.getProperties().getProperty("os.name")+
           ", version: "+System.getProperties().getProperty("os.version")+
	   ", on a "+System.getProperties().getProperty("os.arch"));
    }catch(SecurityException e){
      throw new dSelfException("Sorry, no permission to detect the OS with"+
             "\"OperatingSystem\".",
             "_OperatingSystem", "securityError");
    }     
  }  

 /**
  * Sends the unary message named by the argument to this object, and
  * returns the result of this action. Only ordinary messages can be
  * performed. If a primitive message was sent, then an "primitiveFailedError"
  * error is sent to the "IfFail:"-block, if there is one.
  *
  * @param msg The argument type must be StringSO and is checked by
  * the method
  * @return The result of the performance.
  */
  protected DataSO _Perform(dSelfObject msg) 
              throws dSelfException, NonLocalReturnException{
  
    checkIfStringSO("_Perform", msg);
    
    // Primitves can't be send with _Perform
    if(((StringSO)msg).string.charAt(0) == '_')
      throw new dSelfException("Cannot perform a primitve message \""+
        ((StringSO)msg).string+"\" with \"_Perform\"", "_Perform", "primitiveFailedError"); 
			 
    return dispatchMsg(new OrdinaryMsg(((StringSO)msg).string));
  }
  
 /**
  * Does the same like {@link #_Perform(dSelfObject)}, but for binary
  * and keyword messages. The given vector consits of the name of the
  * message (first element) and the arguments for this message. If too
  * few or too many arguments were sent, then an "primitiveFailedError"
  * error is sent to the "IfFail:"-block, if there is one.
  *
  * @param args The type of the first element must be StringSO and is 
  * checked by the method. The other elements are its arguments.
  * @return The result of the performance.
  */ 
  protected DataSO _PerformWith(Vector args) 
            throws dSelfException, NonLocalReturnException{
  
    // The first element is the name of the message
    checkIfStringSO("_Perform:With:", (dSelfObject)args.firstElement());
    
    String msg = ((StringSO)args.firstElement()).string;
    int num = 0;
     
    // Primitives can't be send with _Perform
    if(msg.charAt(0) == '_')
      throw new dSelfException("Cannot perform a primitve message \""+
                         msg+"\" with \"_PerformWith:\"",
			 "_Perform:With:", "primitiveFailedError"); 
    
    // The message has as much arguments as colons, that 
    // appear in its name 
    for(int i=0; i<msg.length(); i++)
      if(msg.charAt(i) == ':')
        num++;
        
    if(args.size()-1 != num)
      throw new dSelfException("Wrong number of arguments in \""+msg+"\" ! "
            +(args.size()-1)+" arguments were send, but "+num+" are needed.",
	    "_Perform:With:", "primitiveFailedError");

    // Remove the first argument, which was the name of the message
    args.remove(0);
    	    
    return dispatchMsg(new OrdinaryMsg(msg, args));
  }

 /**
  * Resends the unary message named by the argument to this object, and
  * returns the result of this action. Only ordinary messages can be
  * performed. If a primitive message was sent, then an "primitiveFailedError"
  * error is sent to the "IfFail:"-block, if there is one.
  *
  * @param msg The argument type must be StringSO and is checked by
  * the method
  * @return The result of the performance.
  */
  protected DataSO _PerformResend(dSelfObject msg) 
            throws dSelfException, NonLocalReturnException{
  
    checkIfStringSO("_Perform:Resend:", msg);

    // Primitves can't be send with _PerformResend
    if(((StringSO)msg).string.charAt(0) == '_')
      throw new dSelfException("Cannot perform a primitve message \""+
            ((StringSO)msg).string+"\" with \"_Perform:Resend:\"",
	    "_Perform:With:{With:}", "primitiveFailedError");
			 
    return dispatchMsg(new OrdinaryMsg(((StringSO)msg).string, ""));
  }
  
 /**
  * Does the same like {@link #_PerformResend(dSelfObject)}, but for binary
  * and keyword messages. The given vector consits of the name of the
  * message (first element) and the arguments for this message. If too
  * few or too many arguments were sent, then an "primitiveFailedError"
  * error is sent to the "IfFail:"-block, if there is one.
  *
  * @param args The type of the first element must be StringSO and is 
  * checked by the method. The other elements are its arguments.
  * @return The result of the performance.
  */ 
  protected DataSO _PerformResendWith(Vector args)
            throws dSelfException, NonLocalReturnException{
  
    // The first argument is the name of the message
    checkIfStringSO("_PerformResend:With:", (dSelfObject)args.firstElement());

    String msg = ((StringSO)args.firstElement()).string;
    int num = 0;
     
    // Primitves can't be send with _PerformResend
    if(msg.charAt(0) == '_')
      throw new dSelfException("Cannot perform a primitve message \""+
             msg+"\" with \"_PerformResend:With:\"", 
	    "_PerformResend:With:", "primitiveFailedError");
			 
    // The message has as much arguments as colons, that 
    // appear in its name 
    for(int i=0; i<msg.length(); i++)
      if(msg.charAt(i) == ':')
        num++;
        
    if(args.size()-1 != num)
      throw new dSelfException("Wrong number of arguments in \""+msg+"\" ! "
            +(args.size()-1)+" arguments were send, but "+num+" are needed.",
	    "_PerformResend:With:", "primitiveFailedError");

    // Remove the first argument, which was the name of the message
    args.remove(0);
    	    
    return dispatchMsg(new OrdinaryMsg(msg, args, ""));
  }

 /**
  * Delegates the unary message named by the first argument to the object, 
  * that is specified by the second argument and returns the result of this 
  * action. Only ordinary messages can be performed. If a primitive message 
  * was sent, then an "primitiveFailedError" error is sent to the 
  * "IfFail:"-block, if there is one.
  *
  * @param msg1 The argument type must be StringSO and is checked by
  * the method
  * @param msg2 The argument type must be StringSO and is checked by
  * the method
  * @return The result of the performance.
  */
  protected DataSO _PerformDelegatingTo(dSelfObject msg1, dSelfObject msg2) 
            throws dSelfException, NonLocalReturnException{
  
    checkIfStringSO("Perform:DelegatingTo:", msg1);
    checkIfStringSO("Perform:DelegatingTo:", msg2);

    // Primitves can't be send with _PerformResend
    if(((StringSO)msg1).string.charAt(0) == '_')
      throw new dSelfException("Cannot perform a primitve message \""+
          ((StringSO)msg1).string+"\" with \"_PerformDelegatingTo\"",
	  "Perform:DelegatingTo:", "primitiveFailedError"); 
			 
    return dispatchMsg(new OrdinaryMsg(((StringSO)msg1).string, 
                                   ((StringSO)msg2).string));
  }
  
 /**
  * Does the same like {@link #_PerformDelegatingTo(dSelfObject, dSelfObject)}, 
  * but for binary and keyword messages. The given vector consits of 
  * the name of the message (first element), its receiver (second element)
  * and the arguments for this message. If too few or too many arguments 
  * were sent, then an "primitiveFailedError" error is sent to the 
  * "IfFail:"-block, if there is one.
  *
  * @param args The type of the first element must be StringSO and is 
  * checked by the method. The other elements are its arguments.
  * @return The result of the performance.
  */ 
  protected DataSO _PerformDelegatingToWith(Vector args) 
            throws dSelfException, NonLocalReturnException{
  
    // The first argument is the name of the message
    checkIfStringSO("Perform:DelegatingToWith:", (dSelfObject)args.firstElement());
    checkIfStringSO("Perform:DelegatingToWith:", (dSelfObject)args.get(1));

    String msg1 = ((StringSO)args.firstElement()).string;
    String msg2 = ((StringSO)args.get(1)).string;
    int num = 0;
     
    // Primitves can't be send with _PerformResend
    if(msg1.charAt(0) == '_')
      throw new dSelfException("Cannot perform a primitve message \""+
          msg1+"\" with \"_PerformDelegatingToWith:\"",
	  "Perform:DelegatingToWith:", "primitiveFailedError"); 
			 
    // The message has as much arguments as colons, that 
    // appear in its name 
    for(int i=0; i<msg1.length(); i++)
      if(msg1.charAt(i) == ':')
        num++;
        
    if(args.size()-2 != num)
      throw new dSelfException("Wrong number of arguments in \""+msg1+"\" ! "
          +(args.size()-2)+" arguments were send, but "+num+" are needed.",
	  "Perform:DelegatingToWith:", "primitiveFailedError");

    // Remove the first and second arguments, which were the names
    // of the messages 
    args.remove(0);
    args.remove(0);
    	    
    return dispatchMsg(new OrdinaryMsg(msg1, args, msg2));
  }

 /**
  * Prints this object and its contents.
  */ 
  protected NilSO _Print(){

    dSelfVM.printMessage(getName()+": "+getSlotVector().printSlots());
    return new NilSO();
  }  
  
 /**
  * Exits the dSelf virtual machine.
  */
  protected void _Quit(){
  
     dSelfVM.exit(); 
  }  

 /**
  * Returns the number of milliseconds since January 1, 1970, 
  * 00:00:00 GMT
  */
  protected LongSO _TimeReal(){
  
    return new LongSO((new Date()).getTime());
  }  
}     
