package dSelf;
 
import java.util.Vector;
import java.io.*;
import dSelfVM;
 
/**
 * The class IntegerSO repesents the integer numbers of dSelf.
 * IntegerSO is a primitive dSelf object and so it has only one
 * parent slot called "parent", that is shared by all integers.
 * The object, that "parent" refers to has initialy only one
 * parent slot, also called "parent" that refers to the lobby.
 * This object is dedicated to include the user-defined methods,
 * (e.g. +, -, * ...) that are shared by all integers.
 */
public class IntegerSO extends SerializedPrimitiveSO{
 
  /** The value of the integer object */
  public int value = 0;

 /** The shared parent object for all integers */
  protected static LocalOrdinarySO parent = 
    new LocalOrdinarySO(new ParentSlot("parent", dSelfVM.lobbySO));

 /**
  * The shared slots for all integers. It consists of only one slot
  * called "parent"
  */
  protected static SlotVector slotVector = 
    new SlotVector(new ParentSlot("parent",parent));   
    
 /**
  * The shared parent slots for all integers. It consists of only one slot
  * called "parent"
  */
  protected static Vector parVector = new Vector();   
  
  static{
    parVector.add(parent);   
  }

 /**
  * Creates new dSelf integer with the given Java integer value
  */  
  public IntegerSO(int intValue){

    value = intValue;
  }
    
 /**
  * Creates new dSelf integer with the given Java Integer value
  */  
  public IntegerSO(Integer intValue){

    value = intValue.intValue();
  }

 /**
  * Returns the slots of the integers. I.e, it's only one slot
  * called "parent".
  */   
  public SlotVector getSlotVector(){
    
    return slotVector; 
  }
  
 /**
  * Returns the shared parent slot for all dSelf integers.
  */  
  protected LocalOrdinarySO getParent(){
  
    return parent;
  }
    
 /**
  * Returns the parent slots of the integers. I.e, it's only one slot
  * called "parent".
  */   
  public Vector getParentVector(){
  
    return parVector;
  }   

 /**
  * Returns the value of this integer in the form "integer(...)".
  */
  public String getName(){

    return "integer("+value+")";
  }

 /**
  * The dispatcher for dSelf integers, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
      throws dSelfException, NonLocalReturnException{

    try{
    
      switch(msg.getMessageID()){
  
        case PrimMsg.EQ:                     
   	  return _Eq(msg.getFirstArg());
  
        case PrimMsg.INTADD:                     
	  return _IntAdd(msg.getFirstArg());
  	
        case PrimMsg.INTAND:                     
    	  return _IntAnd(msg.getFirstArg());
  	
        case PrimMsg.INTARITHMETICSHIFTRIGHT:
          return _IntArithmeticShiftRight(msg.getFirstArg());
	
        case PrimMsg.INTASDOUBLE:                     
	  return _IntAsDouble();
	
        case PrimMsg.INTASFLOAT:                     
	  return _IntAsFloat();

        case PrimMsg.INTASLONG:                     
	  return _IntAsLong();

        case PrimMsg.INTASSHORT:                     
	  return _IntAsShort();

        case PrimMsg.INTDIV:                     
	  return _IntDiv(msg.getFirstArg());
	
        case PrimMsg.INTEQ:                     
	  return _IntEQ(msg.getFirstArg());
	
        case PrimMsg.INTGE:                     
	  return _IntGE(msg.getFirstArg());
	
        case PrimMsg.INTGT:                     
	  return _IntGT(msg.getFirstArg());
	
        case PrimMsg.INTLE:                     
	  return _IntLE(msg.getFirstArg());
	
        case PrimMsg.INTLT:                     
	  return _IntLT(msg.getFirstArg());
	
        case PrimMsg.INTLOGICALSHIFTLEFT:                     
	  return _IntLogicalShiftLeft(msg.getFirstArg());
	
        case PrimMsg.INTLOGICALSHIFTRIGHT:                     
	  return _IntLogicalShiftRight(msg.getFirstArg());
	
        case PrimMsg.INTMOD:                     
	  return _IntMod(msg.getFirstArg());
	
        case PrimMsg.INTMUL:                     
	  return _IntMul(msg.getFirstArg());
	
        case PrimMsg.INTNE:                     
	  return _IntNE(msg.getFirstArg());
	
        case PrimMsg.INTOR:                     
	  return _IntOr(msg.getFirstArg());
	
        case PrimMsg.INTSUB:                     
	  return _IntSub(msg.getFirstArg());
	
        case PrimMsg.INTXOR:                     
	  return _IntXor(msg.getFirstArg());
      }

      return super.dispatchPrimMsg(msg);       
    
    }catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Checks, if the argument is equal to the value of this integer object.
  *
  * @param arg The argument must be of type IntegerSO
  */
  protected BooleanSO _Eq(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_Eq:", so);
    return BooleanSO.asBooleanSO(((IntegerSO)so).value == value);
  }  

 /**
  * Adds another dSelf integer to this integer
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected IntegerSO _IntAdd(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntAdd:", so);
    return new IntegerSO(value + ((IntegerSO)so).value); 
  }  
  
 /**
  * Returns a bitwise AND of this integer and the argument
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */ 
  protected IntegerSO _IntAnd(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntAnd:", so);  
    return new IntegerSO(value & ((IntegerSO)so).value); 
  }  
  
 /**
  * Arithmetic right shift of this integer by the numbers of bits indicated
  * by the argument. The sign bit is preserved
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected IntegerSO _IntArithmeticShiftRight(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntArithmeticShiftRight:", so);  
    return new IntegerSO(value >> ((IntegerSO)so).value);
  }    
  
 /**
  * Converts this dSelf integer to a dSelf double.  
  */
  protected DoubleSO _IntAsDouble(){
  
    return new DoubleSO((double) value);  
  }  

 /**
  * Converts this dSelf integer to a dSelf float.  
  */
  protected FloatSO _IntAsFloat(){
  
    return new FloatSO((float) value);  
  }  

 /**
  * Converts this dSelf integer to a dSelf long.  
  */
  protected LongSO _IntAsLong(){
  
    return new LongSO((long) value);  
  }  

 /**
  * Converts this dSelf integer to a dSelf short.  
  */
  protected ShortSO _IntAsShort(){
  
    return new ShortSO((short) value);  
  }  

 /**
  * Divides this integer through another integer
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected IntegerSO _IntDiv(dSelfObject so) throws dSelfException{
  
    checkIfIntegerSO("_IntDiv", so);  

    try{  
      return new IntegerSO(value / ((IntegerSO)so).value); 
    }catch(ArithmeticException e){      
      throw new dSelfException(" Division by zero -> "+value+" _IntDiv: 0 !",
              "_IntDiv:", "divisionByZeroError");
    }  
  }  
  
 /**
  * Checks, if the argument is equal to the value of this integer object.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected BooleanSO _IntEQ(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntEQ:", so);  
    return BooleanSO.asBooleanSO(value == ((IntegerSO)so).value);
  }
  
 /**
  * Checks, if this integer is greater or equal than its argument.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected BooleanSO _IntGE(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntGE:", so);  
    return BooleanSO.asBooleanSO(value >= ((IntegerSO)so).value);
  }
      
 /**
  * Checks, if this integer is greater than its argument.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected BooleanSO _IntGT(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntGT:", so);  
    return BooleanSO.asBooleanSO(value > ((IntegerSO)so).value);
  }
  
 /**
  * Checks, if this integer is less or equal than its argument.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected BooleanSO _IntLE(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntLE:", so);  
    return BooleanSO.asBooleanSO(value <= ((IntegerSO)so).value);
  }
  
 /**
  * Checks, if this integer is less than its argument.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected BooleanSO _IntLT(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntLT:", so);  
    return BooleanSO.asBooleanSO(value < ((IntegerSO)so).value);
  }
  
 /**
  * Logical left shift of this integer by the numbers of bits indicated
  * by the argument. 
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected IntegerSO _IntLogicalShiftLeft(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntLogicalShiftLeft:", so);  
    return new IntegerSO(value << ((IntegerSO)so).value);
  }    

 /**
  * Logical right shift of this integer by the numbers of bits indicated
  * by the argument. 0 is shifted into the most significant bit.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected IntegerSO _IntLogicalShiftRight(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntLogicalShiftRight:", so);  
    return new IntegerSO(value >>> ((IntegerSO)so).value);
  }    
  
 /**
  * Returns this integer modulo argument. 
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected IntegerSO _IntMod(dSelfObject so) throws dSelfException{
  
    checkIfIntegerSO("_IntMod:", so);  

    try{  
      return new IntegerSO(value % ((IntegerSO)so).value);  
    }catch(ArithmeticException e){
      throw new dSelfException(" Division by zero -> "+value+" _IntMod: 0 !"
           , "_IntMod:", "divisionByZeroError");
    }  
  }  
  
 /**
  * Returns the product of this integer and the argument. 
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected IntegerSO _IntMul(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntMul:", so);  
    return new IntegerSO(value * ((IntegerSO)so).value); 
  }  
  
 /**
  * Checks if this integer is inequal to its receiver
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */ 
  protected BooleanSO _IntNE(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntNE:", so);  
    return BooleanSO.asBooleanSO(value != ((IntegerSO)so).value);
  }
  
 /**
  * Returns a bitwise OR of this integer and the argument
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */ 
  protected IntegerSO _IntOr(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntOr:", so);  
    return new IntegerSO(value | ((IntegerSO)so).value);
  }  
  
 /**
  * Returns this integer minus the argument.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */
  protected IntegerSO _IntSub(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntSub:", so);  
    return new IntegerSO(value - ((IntegerSO)so).value); 
  }  
  
 /**
  * Returns a bitwise XOR of this integer and the argument
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */ 
  protected IntegerSO _IntXor(dSelfObject so) throws dSelfException{

    checkIfIntegerSO("_IntXor:", so);  
    return new IntegerSO(value ^ ((IntegerSO)so).value);  
  }  
}   
