package dSelf;
import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import javax.swing.event.*;
import javax.swing.tree.*;
import javax.swing.border.TitledBorder;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector; 
import java.util.Enumeration; 
import java.io.*;
import dSelfVM;
import dSelfComp;

/**
 * JPanelFrontEnd represents a GUI for the virtual machine of dSelf.
 * It is designed to be the content of a JFrame.
 */
public class JPanelFrontEnd extends JPanel 
    implements VMFrontEnd, ActionListener{

  /** The frame, to which the JPanel belong to. */
  public static JFrame frame;
  /** The cache for the history feature. */
  protected HistoryCache history;
  /** The text area for the output of the dSelf VM. */
  protected JTextArea outputTextArea;
  /** The text area for the input of the user. */
  protected JTextArea inputTextArea;
  /** The button for entering the input of the input text area. */
  protected JButton enterButton;  
  /** The button for selecting a previous input of the user. */
  protected JButton upButton;  
  /** The button for selecting the next input of the user. */
  protected JButton downButton;  
  /** The button for loading scripts. */
  protected JButton loadButton;  
  /** The button for displaying the slot browser. */
  protected JButton treeButton;  
  /** The button for exiting the dSelf VM. */
  protected JButton exitButton;  
  /** The button for changing the preferences. */
  protected JButton preferenceButton;  
  /** The pane for the buttons of input field. */
  protected JPanel inputButtonsPane;  
  /** The pane for the input field. */
  protected JPanel inputPane;  
  /** The pane for the buttons on the right hand side of the editor. */
  protected JPanel buttonPane;  
  /** A split pane that splits the input and output fields. */
  protected JSplitPane splitTextPanes;  
  /** A pane, that makes the input area scrollable. */
  protected JScrollPane inputScrollPane;  
  /** A pane, that makes the output area scrollable. */
  protected JScrollPane resultTextScrollPane;  
  /** The scroll bar for the output area. */
  protected JScrollBar resultScrollBar;
    
 /**
  * Creates a new front-end.
  */   
  public JPanelFrontEnd(){
  
    super(); 
    
    history = new HistoryCache();
    
    setName("dSelfVM");
    setLayout(new BorderLayout()); 

    outputTextArea = new JTextArea("", 20, 60);
    outputTextArea.setFont(new Font("Courier", Font.PLAIN, 10));
    outputTextArea.setBackground(Color.lightGray);
    outputTextArea.setForeground(Color.black);
    outputTextArea.setLineWrap(true);
    outputTextArea.setWrapStyleWord(true);
    outputTextArea.setEditable(false);

    resultTextScrollPane = new JScrollPane(outputTextArea,
          JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, 
	  JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);

    resultScrollBar = resultTextScrollPane.getVerticalScrollBar();

    inputTextArea = new JTextArea("", 5, 60);
    inputTextArea.setFont(new Font("Courier", Font.PLAIN, 12));
    inputTextArea.setLineWrap(false);
    inputTextArea.setWrapStyleWord(false);
    
    inputScrollPane = new JScrollPane(inputTextArea,
	  JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
          JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
    
    enterButton = new JButton(new ImageIcon("images/enter.gif"));
    enterButton.setToolTipText("Evaluate the expression");
    enterButton.setMnemonic(KeyEvent.VK_ENTER);
    enterButton.setActionCommand("enter");
    enterButton.addActionListener(this);

    upButton = new JButton(new ImageIcon("images/arrow_up.gif"));
    upButton.setToolTipText("Previous input");
    upButton.setMnemonic(KeyEvent.VK_UP);
    upButton.setActionCommand("previous");
    upButton.addActionListener(this);

    downButton = new JButton(new ImageIcon("images/arrow_down.gif"));
    downButton.setToolTipText("Next input");
    downButton.setMnemonic(KeyEvent.VK_DOWN);
    downButton.setActionCommand("next");
    downButton.addActionListener(this);

    inputButtonsPane = new JPanel();
    inputButtonsPane.setLayout(new BoxLayout(inputButtonsPane, BoxLayout.Y_AXIS));
    inputButtonsPane.add(upButton);
    inputButtonsPane.add(enterButton);
    inputButtonsPane.add(downButton);

    inputPane = new JPanel();   
    inputPane.setLayout(new BoxLayout(inputPane, BoxLayout.X_AXIS));
    inputPane.add(inputButtonsPane);
    inputPane.add(inputScrollPane);
    
    splitTextPanes = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
                                 resultTextScrollPane, inputPane);

    treeButton = new JButton(new ImageIcon("images/tree.gif")); 
    treeButton.setActionCommand("tree");
    treeButton.addActionListener(this);
    treeButton.setToolTipText("Browse all slots as a tree");

    preferenceButton = new JButton(new ImageIcon("images/preference.gif"));        
    preferenceButton.setMnemonic(KeyEvent.VK_P);
    preferenceButton.setActionCommand("preference");
    preferenceButton.addActionListener(this);
    preferenceButton.setToolTipText("Preference settings");

    loadButton = new JButton(new ImageIcon("images/disk.gif"));
    loadButton.setMnemonic(KeyEvent.VK_L);
    loadButton.setActionCommand("load");
    loadButton.addActionListener(this);
    loadButton.setToolTipText("Load a dSelf-file");
    
    exitButton = new JButton(new ImageIcon("images/exit.gif"));     
    exitButton.setMnemonic(KeyEvent.VK_Q);
    exitButton.setActionCommand("exit");
    exitButton.addActionListener(this);
    exitButton.setToolTipText("Exit");

    buttonPane = new JPanel();
    buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.Y_AXIS));
    buttonPane.add(treeButton);
    buttonPane.add(loadButton);
    buttonPane.add(preferenceButton);
    buttonPane.add(exitButton);
    
    add(buttonPane, BorderLayout.EAST);
    add(splitTextPanes, BorderLayout.CENTER);       
  }  

 /**
  * Performs the actions, caused by the buttons.
  */ 
  public void actionPerformed(ActionEvent e) {

    if(e.getActionCommand().equals("enter")){

      enterInput();

    }else if(e.getActionCommand().equals("previous")){

      inputTextArea.setText(history.getPrevious());
      inputTextArea.requestFocus();  

    }else if(e.getActionCommand().equals("next")){

      inputTextArea.setText(history.getNext());
      inputTextArea.requestFocus();  

    }else if(e.getActionCommand().equals("exit")){ 

      dSelfVM.exit();

    }else if(e.getActionCommand().equals("preference")){ 

      JFrame preferenceFrame = new Preferences();
      preferenceFrame.pack();
      preferenceFrame.setVisible(true);

    }else if(e.getActionCommand().equals("tree")){ 

      SlotTree slotTree = new SlotTree(frame);
      slotTree.pack();
      slotTree.setVisible(true);

    }else if(e.getActionCommand().equals("load")){ 
      
      loadFile();
    }  
  }     	 
  
 /**
  * Opens a file chooser dialog and loads the script, that was
  * chosen by the user.
  */ 
  protected void loadFile(){
  
    JFileChooser chooser = new JFileChooser();
    chooser.setFileFilter(new dSelfFilter());
    int returnVal = chooser.showOpenDialog(this);
    DataSO dSO = null;
        
    if(returnVal == JFileChooser.APPROVE_OPTION) 
      parseString("'" + chooser.getSelectedFile().getAbsolutePath()
	            + "' _RunScript");
  }

 /**
  * Reads the current input of the input text area and forwards its
  * content to {@link #enterInput()}.
  */
  protected void enterInput(){
    
    String input = inputTextArea.getText();
   
    parseString(input) ;
    
    inputTextArea.setText("");
    inputTextArea.requestFocus();  
        
    history.addHistory(input);
  }
  
 /**
  * Prints the input of the user with a leading "#" on the output
  * text area and parses its content by calling {@link dSelfVM#evalInput(String)}.
  * The result will also be printed on the output text area.
  */ 
  protected void parseString(String input){
  
    outputTextArea.append("# ");

    for(int i=0; i<input.length(); i++)
      if(input.charAt(i) == '\n')
        outputTextArea.append("\n# ");
      else    
        outputTextArea.append("" + input.charAt(i));

    outputTextArea.append("\n");

    String result = dSelfVM.evalInput(input);

    if(result != null && !result.equals(""))
      printMessage("-> "+result+"\n");
      
    scrollToBottom();
  }
 
 /**
  * Welcomes the user with the message "Welcome to dSelf !".
  */   
  public void Run(Vector filenames){
  
    if (dSelfVM.verboseMode)
      printMessage("Welcome to dSelf !\n");
    
    for (int i=0; i<filenames.size(); i++)
      parseString("'" + (String)filenames.get(i) + "' _RunScript");
  }
  
 /**
  * Prints a message on the output text area. 
  */ 
  public void printMessage(String msg){

    outputTextArea.append(msg + "\n");
    scrollToBottom();
  }

 /**
  * Scrolls the output text area down to the bottom.
  */
  protected void scrollToBottom(){
  
    // Add a adjustmentlistener, that scrolls the textarea
    // down, when the scrollbar changed. After doing this, it
    // removes itself, because its work is done. A simple
    // resultScrollBar.setValue(resultScrollBar.getMaximum());
    // directly after adding some text didn't work, because 
    // the value of the scrollbar often changed too late, so 
    // getMaximum() returned the old value and nothing happend. 
    AdjustmentListener adjListener = new AdjustmentListener(){
      public void adjustmentValueChanged(AdjustmentEvent e){
        resultScrollBar.setValue(resultScrollBar.getMaximum());
	resultScrollBar.removeAdjustmentListener(this);
      }
    };
    
    resultScrollBar.addAdjustmentListener(adjListener);    
  }


/*+--------------------------------------------------------------+*/  
/*|                     Inner classes                            |*/
/*+--------------------------------------------------------------+*/  

 /**
  * dSelfFilter is a filter for the file chooser dialog.
  */ 
  protected class dSelfFilter extends FileFilter{
  
   /**
    * Accepts directories and files ending with dSelfComp.dSelfSuffix.
    */
    public boolean accept(File f){

      return f.isDirectory() || f.getName().endsWith(dSelfComp.dSelfSuffix);
    }
    
   /**
    * Returns the desciption for this file filter (i.e. "dSelf files").
    */   
    public String getDescription(){ 
    
      return "dSelf files";
    }
  }
  
 /**
  * HistoryCache realizes a cache for the last inputs of the user.
  */ 
  protected class HistoryCache extends Vector{
  
    /** The number of inputs, that shall be rememberd. */
    protected int historyCacheSize = 100;
    /** An iterator for iterating over all entries. */
    protected int iterator = 0;
    
   /**
    * Changes the size for this cache.
    *
    * @param newCache The new cache size (number of entries).
    */
    public void setHistoryCache(int newCache){
    
      while(this.size() > newCache){
        this.remove(0);
      }
        
      historyCacheSize = newCache;

      iterator = this.size() -1;
    }
    
   /**
    * The current size of this cache.
    */ 
    public int getHistoryCacheSize(){
    
      return historyCacheSize;
    }
   
   /**
    * Adds a new input to the cache. If the cache is full, the oldest
    * input is removed.
    *
    * @param input The new input
    */        
    public void addHistory(String input){
    
      add(input);
      
      if(this.size() >= historyCacheSize)
        this.remove(0);
	
      iterator = this.size();	
    }
    
   /**
    * Moves the iterator to the entry, that occures before the current
    * entry, if there is any.
    *
    * @return The input at the new position
    */ 
    public String getPrevious(){
    
      if(iterator == 0)
        return this.size() == 0 ? "" : (String) get(0);
	
      return (String) get(--iterator);	
    }
    
   /**
    * Moves the iterator to the entry, that occures after the current
    * entry, if there is any.
    *
    * @return The input at the new position
    */ 
    public String getNext(){
    
      if(iterator >= this.size()-1){
        iterator = this.size();
        return "";
      }
      	
      return (String) get(++iterator);
    }  	
  }  
	    
 /**
  * Preferences is a frame, that offers options for configurating the
  * this front-end to the user. 
  */	    
  protected class Preferences extends JFrame implements ActionListener{

    /** A pane to display the settings for the output text area. */
    protected JPanel prefOutputPane;
    /** A pane with the buttons for changing the setting of the output text area. */
    protected JPanel prefOutputButtonPane;
    /** The text area with an example text for the text output area. */
    protected JTextArea prefOutputTextArea;
    /** A button for changing the background color of the text output area. */
    protected JButton prefOutputBackgroundButton;
    /** A button for changing the font of the text output area. */
    protected JButton prefOutputFontButton;
    /** A button for changing the foreground color of the text output area. */
    protected JButton prefOutputForegroundButton;
    /** A pane to display the settings for the input text area. */
    protected JPanel prefInputPane;
    /** A pane with the buttons for changing the setting of the input text area. */
    protected JPanel prefInputButtonPane;
    /** The text area with an example text for the text input area. */
    protected JTextArea prefInputTextArea;
    /** A button for changing the background color of the text input area. */
    protected JButton prefInputBackgroundButton;
    /** A button for changing the font of the text input area. */
    protected JButton prefInputFontButton;
    /** A button for changing the foreground color of the text input area. */
    protected JButton prefInputForegroundButton;
    /** A pane for the history  */
    protected JPanel prefHistoryPane;
    /** A label for the history */
    protected JLabel historyLabel;
    /** A text field for the new history size */
    protected JTextField historyInput;
    /** A pane for the buttons at the bottom of this frame (ok, cancel, ...) */
    protected JPanel buttonPane;
    /** A button for accepting the changes */
    protected JButton okButton; 
    /** A button for applying the changes */
    protected JButton applyButton; 
    /** A button for canceling the changes */
    protected JButton cancelButton; 
    /** A button for loading preferences from a file */
    protected JButton loadButton; 
    /** A button for saving preferences to a file */
    protected JButton saveButton; 
    /** The new size of the history cache */     
    protected int newHistoryCache;

   /** 
    * Creates a new preference frame.
    */
    public Preferences(){

      super("Preference settings");
      
      getContentPane().setLayout(new BoxLayout(getContentPane(), 
                                           BoxLayout.Y_AXIS));

      prefOutputTextArea = new JTextArea(
             "\n This is an example \n for the output area \n"+
	     " of the dSelf-VM. \n");
      prefOutputTextArea.setFont(outputTextArea.getFont());
      prefOutputTextArea.setBackground(outputTextArea.getBackground());
      prefOutputTextArea.setForeground(outputTextArea.getForeground());
      prefOutputTextArea.setEditable(false);

      prefOutputBackgroundButton = new JButton("Background color");     
      prefOutputBackgroundButton.setActionCommand("background color output");
      prefOutputBackgroundButton.addActionListener(this);
      prefOutputBackgroundButton.setToolTipText(
                  "Change the background color for the output area");

      prefOutputForegroundButton = new JButton("Foreground color");     
      prefOutputForegroundButton.setActionCommand("foreground color output");
      prefOutputForegroundButton.addActionListener(this);
      prefOutputForegroundButton.setToolTipText(
                  "Change the foreground color for the output area");

      prefOutputFontButton = new JButton("Font");     
      prefOutputFontButton.setActionCommand("font output");
      prefOutputFontButton.addActionListener(this);
      prefOutputFontButton.setToolTipText(
                  "Change the font for the output area");

      prefOutputButtonPane = new JPanel();
      prefOutputButtonPane.setLayout(
                 new BoxLayout(prefOutputButtonPane, BoxLayout.Y_AXIS));
      prefOutputButtonPane.add(prefOutputBackgroundButton);
      prefOutputButtonPane.add(prefOutputForegroundButton);
      prefOutputButtonPane.add(prefOutputFontButton);

      prefOutputPane = new JPanel();
      prefOutputPane.setBorder(new TitledBorder("Output area text"));
      prefOutputPane.add(prefOutputTextArea, BorderLayout.CENTER);
      prefOutputPane.add(prefOutputButtonPane, BorderLayout.EAST);

      prefInputTextArea = new JTextArea(
             "\n This is an example \n for the input area \n"+
	     " of the dSelf-VM. \n");
      prefInputTextArea.setFont(inputTextArea.getFont());
      prefInputTextArea.setBackground(inputTextArea.getBackground());
      prefInputTextArea.setForeground(inputTextArea.getForeground());
      prefInputTextArea.setEditable(false);

      prefInputBackgroundButton = new JButton("Background color");     
      prefInputBackgroundButton.setActionCommand("background color input");
      prefInputBackgroundButton.addActionListener(this);
      prefInputBackgroundButton.setToolTipText(
                  "Change the background color for the input area");

      prefInputForegroundButton = new JButton("Foreground color");     
      prefInputForegroundButton.setActionCommand("foreground color input");
      prefInputForegroundButton.addActionListener(this);
      prefInputForegroundButton.setToolTipText(
                  "Change the foreground color for the input area");

      prefInputFontButton = new JButton("Font");     
      prefInputFontButton.setActionCommand("font input");
      prefInputFontButton.addActionListener(this);
      prefInputFontButton.setToolTipText(
                  "Change the font for the input area");

      prefInputButtonPane = new JPanel();
      prefInputButtonPane.setLayout(
                 new BoxLayout(prefInputButtonPane, BoxLayout.Y_AXIS));
      prefInputButtonPane.add(prefInputBackgroundButton);
      prefInputButtonPane.add(prefInputForegroundButton);
      prefInputButtonPane.add(prefInputFontButton);

      prefInputPane = new JPanel();
      prefInputPane.setBorder(new TitledBorder("Input area text"));
      prefInputPane.add(prefInputTextArea, BorderLayout.CENTER);
      prefInputPane.add(prefInputButtonPane, BorderLayout.EAST);

      okButton = new JButton("Ok");
      okButton.setActionCommand("ok");
      okButton.addActionListener(this);

      applyButton = new JButton("Apply");
      applyButton.setActionCommand("apply");
      applyButton.addActionListener(this);

      cancelButton = new JButton("Cancel");
      cancelButton.setActionCommand("cancel");
      cancelButton.addActionListener(this);

      loadButton = new JButton("Load");
      loadButton.setActionCommand("load");
      loadButton.addActionListener(this);

      saveButton = new JButton("Save");
      saveButton.setActionCommand("save");
      saveButton.addActionListener(this);

      buttonPane = new JPanel();
      buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.X_AXIS));
      buttonPane.add(okButton);         
      buttonPane.add(applyButton);         
      buttonPane.add(cancelButton);         
      buttonPane.add(loadButton);         
      buttonPane.add(saveButton);         

      newHistoryCache = history.getHistoryCacheSize();

      historyLabel = new JLabel("History size:");
      
      historyInput = new JTextField(
          (new Integer(history.getHistoryCacheSize())).toString());

      historyInput.addActionListener(new ActionListener(){
        public void actionPerformed(ActionEvent ae){
          try{
            newHistoryCache = Integer.parseInt(ae.getActionCommand());
	  }catch(NumberFormatException e){
	    historyInput.setText((new Integer(newHistoryCache)).toString());
	  }  
	}
      }); 

      prefHistoryPane = new JPanel();
      prefHistoryPane.setBorder(new TitledBorder("History"));
      prefHistoryPane.setLayout(new BoxLayout(prefHistoryPane, BoxLayout.X_AXIS));
      prefHistoryPane.add(historyLabel);
      prefHistoryPane.add(historyInput);
            
      getContentPane().add(prefOutputPane);
      getContentPane().add(prefInputPane);
      getContentPane().add(prefHistoryPane);
      getContentPane().add(buttonPane);
    }

   /**
    * Performs the actions, caused by the buttons.
    */ 
    public void actionPerformed(ActionEvent e) {

      if(e.getActionCommand().equals("background color output")){

        prefOutputTextArea.setBackground(
	  JColorChooser.showDialog(this, 
	        "New background color for the output area", 
	        prefOutputTextArea.getBackground()));

      }else if(e.getActionCommand().equals("foreground color output")){

        prefOutputTextArea.setForeground(
	  JColorChooser.showDialog(this, 
	        "New foreground color for the output area", 
	        prefOutputTextArea.getForeground()));

      }else if(e.getActionCommand().equals("font output")){

	prefOutputTextArea.setFont((new FontChooser(this, 
	         prefOutputTextArea.getFont())).getChosenFont());
        pack();

      }else if(e.getActionCommand().equals("background color input")){

        prefInputTextArea.setBackground(
	  JColorChooser.showDialog(this, 
	        "New background color for the input area", 
	        prefInputTextArea.getBackground()));

      }else if(e.getActionCommand().equals("foreground color input")){

        prefInputTextArea.setForeground(
	  JColorChooser.showDialog(this, 
	        "New foreground color for the input area", 
	        prefInputTextArea.getForeground()));

      }else if(e.getActionCommand().equals("font input")){

	prefInputTextArea.setFont((new FontChooser(this, 
	        prefInputTextArea.getFont())).getChosenFont());
        pack();

      }else if(e.getActionCommand().equals("ok")){

	inputTextArea.setFont(prefInputTextArea.getFont());
	outputTextArea.setFont(prefOutputTextArea.getFont());
	inputTextArea.setBackground(prefInputTextArea.getBackground());
	outputTextArea.setBackground(prefOutputTextArea.getBackground());
	inputTextArea.setForeground(prefInputTextArea.getForeground());
	outputTextArea.setForeground(prefOutputTextArea.getForeground());
        try{
	  // Use the latest input of the user if possible
   	  history.setHistoryCache(Integer.parseInt(historyInput.getText()));
	}catch(NumberFormatException ne){
	  history.setHistoryCache(newHistoryCache);
	}  
	
        dispose();        

      }else if(e.getActionCommand().equals("apply")){

	inputTextArea.setFont(prefInputTextArea.getFont());
	outputTextArea.setFont(prefOutputTextArea.getFont());
	inputTextArea.setBackground(prefInputTextArea.getBackground());
	outputTextArea.setBackground(prefOutputTextArea.getBackground());
	inputTextArea.setForeground(prefInputTextArea.getForeground());
	outputTextArea.setForeground(prefOutputTextArea.getForeground());
        try{
	  // Use the latest input of the user if possible
   	  history.setHistoryCache(Integer.parseInt(historyInput.getText()));
	}catch(NumberFormatException ne){
	  history.setHistoryCache(newHistoryCache);
	  historyInput.setText((new Integer(newHistoryCache)).toString());
	}  

      }else if(e.getActionCommand().equals("cancel")){

        dispose();        

      }else if(e.getActionCommand().equals("load")){
      
        JFileChooser chooser = new JFileChooser();
        chooser.setFileFilter(new PreferencesFilter());
        int returnVal = chooser.showOpenDialog(this);
        
        if(returnVal == JFileChooser.APPROVE_OPTION) 
          loadPreferences(chooser.getSelectedFile().getAbsolutePath());
	
      }else if(e.getActionCommand().equals("save")){

        JFileChooser chooser = new JFileChooser();
        chooser.setFileFilter(new PreferencesFilter());
        int returnVal = chooser.showSaveDialog(this);
        
        if(returnVal == JFileChooser.APPROVE_OPTION) 
          savePreferences(chooser.getSelectedFile().getAbsolutePath());
      }        
    }

   /**
    * PreferencesFilter is a filter for the file chooser dialog.
    */ 
    protected class PreferencesFilter extends FileFilter{
  
     /**
      * Accepts directories and files ending with ".dSelfVMPref".
      */
      public boolean accept(File f){

        return f.isDirectory() || f.getName().endsWith(".dSelfVMPref");
      }
      
     /**
      * Returns the desciption for this file filter (i.e. "Preferences files").
      */   
      public String getDescription(){ 
    
        return "Preferences files";
      }
    }

   /**
    * Loads the preferences from a file with the specified path.
    */
    protected void loadPreferences(String path){
	
      try{
        FileInputStream file = new FileInputStream(path);
        ObjectInputStream p = new ObjectInputStream(file);
      
        prefOutputTextArea.setBackground((Color)p.readObject());
        prefOutputTextArea.setForeground((Color)p.readObject());
        prefOutputTextArea.setFont((Font)p.readObject());
        prefInputTextArea.setBackground((Color)p.readObject());
        prefInputTextArea.setForeground((Color)p.readObject());
        prefInputTextArea.setFont((Font)p.readObject());
	history.setHistoryCache(p.readInt());
	p.close();
      }
      catch(IOException e){
        System.err.println("I/O-Error: "+e.getMessage());
      }      
      catch(ClassNotFoundException e){
        System.err.println("Class-Error: "+e.getMessage());
      }      
    }

   /**
    * Saves the preferences to a file with the specified path.
    */
    protected void savePreferences(String path){
      
      if(!path.endsWith(".dSelfVMPref"))
        path += ".dSelfVMPref";       

      try{
        FileOutputStream file = new FileOutputStream(path);
        ObjectOutputStream p = new ObjectOutputStream(file);
      
        p.writeObject(prefOutputTextArea.getBackground());
        p.writeObject(prefOutputTextArea.getForeground());
        p.writeObject(prefOutputTextArea.getFont());
        p.writeObject(prefInputTextArea.getBackground());
        p.writeObject(prefInputTextArea.getForeground());
        p.writeObject(prefInputTextArea.getFont());
	p.writeInt(history.getHistoryCacheSize());
	p.close();
      }
      catch(IOException e){
        System.err.println("I/O-Error: "+e.getMessage());
      }      
    }
    
   /**
    * FontChooser is a dialog, that lets the user choose a new
    * font for the input or output text area.
    */ 
    protected class FontChooser extends JDialog  
                    implements ItemListener, ActionListener{

      /** The font, that was chosen by the user. */
      protected Font chosenFont; 
      /** The font, that was displayed before the user chose a new one. */ 
      protected Font oldFont; 
      /** The name of the font family. */
      protected String familyName;
      /** The style of the font. */
      protected int style;
      /** The size of the font. */
      protected int size;
      /** A text area that displays an example text to demonstrate the changes. */
      protected JTextArea exampleTextArea;
     /** 
      * A scroll pane, that generates a scroll bar, when the size of
      * the example text with the chosen font gets bigger than the 
      * frame size.
      */ 
      protected JScrollPane fontScrollPane;
      /** A pane for the buttons. */
      protected JPanel buttonPane;
      /** A pane for the combo boxes. */
      protected JPanel fontBoxesPane;
      /** A button for accepting the changes. */
      protected JButton okButton;
      /** A button for canceling the changes. */
      protected JButton cancelButton;
      /** A combo box with the names of the available font families. */
      protected JComboBox familyComboBox;
      /** A combo box with the possible sizes for the font. */
      protected JComboBox sizeComboBox;
      /** A combo box with the possible styles for the font. */
      protected JComboBox styleComboBox;

     /**
      * Creates a new dialog for choosing a font.
      *
      * @param parent The parent frame, that called this dialog
      * @param oldFont The actually set font, that shall be changed 
      */
      public FontChooser(Frame parent, Font oldFont){
     
        super(parent, "Choose a font", true);
     
        this.oldFont = chosenFont = oldFont;
        familyName = oldFont.getFamily();
        style = oldFont.getStyle();
        size = oldFont.getSize();
          
        exampleTextArea = new JTextArea(
          "\n The Quick Brown Fox Jumps Over The Lazy Dog. \n\n"+
	  " _AddSlots: (|\n"+
	  "   x <- 1.\n"+
	  "   ++ = (| :arg | x + arg)\n"+
	  " |)\n");
        exampleTextArea.setFont(chosenFont);
        exampleTextArea.setEditable(true);  	
	
        fontScrollPane = new JScrollPane(exampleTextArea,
	    JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
    
        okButton = new JButton("Ok");
        okButton.setActionCommand("ok");
        okButton.addActionListener(this);

        cancelButton = new JButton("Cancel");
        cancelButton.setActionCommand("cancel");
        cancelButton.addActionListener(this);
      
        buttonPane = new JPanel();
        buttonPane.add(okButton);
        buttonPane.add(cancelButton);
    
        GraphicsEnvironment gEnv = GraphicsEnvironment.getLocalGraphicsEnvironment();
        String envFonts[] = gEnv.getAvailableFontFamilyNames();

        // Create a combo box with all font family names and
	// set the currently used font as selected. The comparison
	// isn't case sensitive,
        familyComboBox = new JComboBox(envFonts);
        familyComboBox.addItemListener(this);
        for(int i=0; i<familyComboBox.getItemCount(); i++){
          if(((String)familyComboBox.getItemAt(i))
	     .equalsIgnoreCase(familyName)){
	    familyComboBox.setSelectedIndex(i);
	    break; 
	  }  
        }
      
        sizeComboBox = new JComboBox(new Object[]
          { "8", "10", "12", "14", "16", "18", "20", "22", "24", "26", "28"});
        sizeComboBox.addItemListener(this);
        for(int i=0; i<sizeComboBox.getItemCount(); i++){
          if((new Integer((String)sizeComboBox.getItemAt(i))).equals(new Integer(size))){
	    sizeComboBox.setSelectedIndex(i);
	    break; 
	  }  
        }

        styleComboBox  = new JComboBox(new Object[]
              { "PLAIN", "BOLD", "ITALIC", "BOLD & ITALIC"});
        styleComboBox.addItemListener(this);
        styleComboBox.setSelectedIndex(style == Font.PLAIN ? 0 :
                                   style == Font.BOLD ? 1 :
                                   style == Font.ITALIC ? 2 : 3);

        fontBoxesPane = new JPanel();
        fontBoxesPane.setLayout(new BoxLayout(fontBoxesPane, BoxLayout.X_AXIS));
        fontBoxesPane.add(familyComboBox); 
        fontBoxesPane.add(sizeComboBox); 
        fontBoxesPane.add(styleComboBox); 

        this.getContentPane().add(fontBoxesPane, BorderLayout.NORTH);
        this.getContentPane().add(fontScrollPane, BorderLayout.CENTER);
        this.getContentPane().add(buttonPane, BorderLayout.SOUTH);
      }
      
     /**
      * Returns the font, that was chosen by the user.
      */ 
      public Font getChosenFont(){
    
        this.pack();
        this.show();
        return chosenFont;
      }
     
     /**
      * This method is called, when another item of the combo boxes
      * was selected.
      */
      public void itemStateChanged(ItemEvent e) {

        if(e.getStateChange() != ItemEvent.SELECTED) 
          return;
        
        Object list = e.getSource();

        if(list == familyComboBox){

          familyName = (String)familyComboBox.getSelectedItem();

        }else if(list == sizeComboBox){

          size = Integer.parseInt(((String)sizeComboBox.getSelectedItem()));

        }else if(list == styleComboBox){

          String styleStr = (String)styleComboBox.getSelectedItem();
          style = styleStr.equals("PLAIN") ? Font.PLAIN : 
	          styleStr.equals("BOLD") ? Font.BOLD : 
	          styleStr.equals("ITALIC") ? Font.ITALIC : 
	          Font.BOLD | Font.ITALIC;  
        }     
	
        chosenFont = new Font(familyName, style, size); 
        exampleTextArea.setFont(chosenFont); 	
      }	        

     /**
      * Performs the actions, caused by the buttons.
      */ 
      public void actionPerformed(ActionEvent e) {
      
        if(e.getActionCommand().equals("ok")){
          this.dispose();
        }else if(e.getActionCommand().equals("cancel")){
          chosenFont = oldFont;
	  this.dispose();
        }
      }        
    }
  }
  
 /**
  * SlotTree represents a slot browser for the slots of dSelf.
  */ 
  protected class SlotTree extends JDialog implements ActionListener{
  
    /** The root node of the tree. */
    protected DefaultMutableTreeNode root;
    /** The model of the tree. */
    protected DefaultTreeModel treeModel;
    /** Displays the tree. */
    protected JTree tree;
    /** A scroll pane, if the tree gets larger than the visible area. */
    protected JScrollPane treeScrollPane;
    /** The button, that resets the slot browser. */
    protected JButton button;
    /** The depth of this tree */
    protected int treeDepth = 5;
    
    private void createTree(DefaultMutableTreeNode node, DataSO dSO, int depth){
    
      if(depth == 0) return;
   
      SlotVector slots = dSO.getSlotVector();
      Slot slot = null;
   
      for(int j=0; j<slots.size(); j++){
       slot = slots.getSlotAt(j);
       if(slot instanceof MethodSlotImpl){
         node.add(new DefaultMutableTreeNode(slot.getName() + " (method)"));
       }else if(slot instanceof AssignmentSlot){
         node.add(new DefaultMutableTreeNode(slot.getName() + " <-"));
       }else{  
         DefaultMutableTreeNode newNode = 
           new DefaultMutableTreeNode(slot.getName());
         node.add(newNode);
         createTree(newNode, (DataSO)slot.getContent(), depth-1);
       }  
     }  
    }
    
   /**
    * Creates a new slot browser.
    */ 
    public SlotTree(JFrame parent){
      
      super(parent, "Slot browser", true);
	     
      DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
      ImageIcon icon= new ImageIcon("images/slot.gif");
      renderer.setLeafIcon(icon);
      renderer.setOpenIcon(icon);
      renderer.setClosedIcon(icon);

      root = new DefaultMutableTreeNode("lobby"); 
      createTree(root, dSelfVM.lobbySO, treeDepth);
      treeModel = new DefaultTreeModel(root);
      tree = new JTree(treeModel);
      tree.setCellRenderer(renderer);   
      tree.setShowsRootHandles(true);
 
      treeScrollPane = new JScrollPane(tree);
      getContentPane().add(treeScrollPane, BorderLayout.CENTER);
    
      button = new JButton("Change Depth");
      button.setMnemonic(KeyEvent.VK_C);
      button.setActionCommand("reset");
      button.addActionListener(this);
      button.setToolTipText("Reset the tree");
      getContentPane().add(button, BorderLayout.SOUTH);
     
    }  

   /**
    * Performs the actions, caused by the reset-button.
    */ 
    public void actionPerformed(ActionEvent e) {

      if(e.getActionCommand().equals("reset"))
        reset();
    }
    
   /**
    * Resets the slot browser and shows changes. 
    */ 
    public void reset(){
      
      String depStr = JOptionPane.showInputDialog(this, "Enter new Depth");    
      int dep = treeDepth;
      try{
        dep = Integer.parseInt(depStr);
      }catch(NumberFormatException e){
        JOptionPane.showMessageDialog(this, "Value must be a number !",
	  "Error", JOptionPane.ERROR_MESSAGE);
        return;
      }
      
      if(dep<5){
        JOptionPane.showMessageDialog(this, "Value must be greater than 4 !",
	  "Error", JOptionPane.ERROR_MESSAGE);
      }else{	   
        treeDepth = dep;  	
        root.removeAllChildren();
        createTree(root, dSelfVM.lobbySO, treeDepth);
        treeModel.reload();
      }	
    }
  }
}

