package dSelf;

import java.rmi.*;
import java.net.MalformedURLException;

/**
 * OrdinarySO represents the ordinary (non primitive) data objects
 * of dSelf. Data objects consist of a number of slots, or are empty.
 */
public abstract class OrdinarySO extends DataSO implements RemoteReferenceSO{

  /** The annotation of this object */
  protected String annotation = null;

  /** Returns the annotation for this object */
  protected String getAnnotation(){ 
    
    return annotation == null ? "No annotation available." : annotation;
  }  
  
 /**
  * Removes all slots of this objects.
  */ 
  protected abstract void removeAllSlots();

 /**
  * Removes the slot with the specified name. When no slot with
  * such name is found an exception is thrown
  *
  * @param remSlot The name of the slot
  */ 
  protected abstract void removeSlot(String remSlot) throws dSelfException;  

 /**
  * Adds some slots to this object. If the given slot vector
  * has one or more slots with names, that already exist in this 
  * object, then the old slots with the same name will be removed.
  *
  * @param slots The slots, that will be added
  */
  protected abstract void addSlots(SlotVector slots) throws dSelfException;

 /**
  * Adds some slots to this object. If the given slot vector
  * has one or more slots with names, that already exist in this 
  * object, then the new slots with the same name won't be added.
  *
  * @param slots The slots, that will be added
  */
  protected abstract void addSlotsIfAbsent(SlotVector slots) 
      throws dSelfException;
  
 /**
  * Returns a clone of this object vector.
  */
  public Object clone(){
  
    return new LocalOrdinarySO((SlotVector) getSlotVector().clone(),
        annotation);
  }

 /**
  * Check if the given slot name is a parent slot and has the 
  * right syntax
  *
  * @param slotName The name of the slot
  * @param msgName The name of the message, that caused an error,
  * when the slot wasn't found
  */
  protected boolean isParentSlot(String slotName, String msgName)
      throws dSelfException{

    // Slot names must begin with a small letter or underscore 
    if(!isCorrectStartChar(slotName.charAt(0)))
      throw new dSelfException(
         "The identifier \"" + slotName + "\"must begin with a small letter or"
	 +" an underscore to be a correct slot name !",
	 msgName, "badTypeError");
 
    // Check the body of the slot name   
    for(int i=1; i<slotName.length()-1; i++)
      if(!isCorrectBodyChar(slotName.charAt(i)))
        throw new dSelfException(
          "Illegal character in body of identifier \"" + slotName + "\" !",
	  msgName, "badTypeError");
      
    // Check the last character. If it is a star, then it is a parent
    // slot, otherwise not.       
    char ch = slotName.charAt(slotName.length()-1);

    if(ch == '*'){
      return true;
    }else if(isCorrectBodyChar(ch)){   
      return false;
    }else{
      throw new dSelfException(
        "Illegal character in body of identifier \"" + slotName + "\" !",
	msgName, "badTypeError");
    }      
  }

 /**
  * Check if the given character can be the first character
  * in a slot name
  */ 
  private boolean isCorrectStartChar(char ch){
  
    int val_a = Character.getNumericValue('a');
    int val_z = Character.getNumericValue('z');
    int chVal = Character.getNumericValue(ch);
  
    return ((ch == '_') || (chVal >= val_a && chVal <= val_z));
  }
    
 /**
  * Check if the given character can occur in the body of a
  * slot name
  */ 
  private boolean isCorrectBodyChar(char ch){
  
    int val_a = Character.getNumericValue('a');
    int val_z = Character.getNumericValue('z');
    int val_A = Character.getNumericValue('A');
    int val_Z = Character.getNumericValue('Z');
    int chVal = Character.getNumericValue(ch);

    // The body of slot name consists of either a letter, digit
    // or underscore
    return ((chVal >= val_a && chVal <= val_z) || 
        (chVal >= val_A && chVal <= val_Z) || 
	(ch == '_') || Character.isDigit(ch));
  }
  
 /**
  * The dispatcher for dSelf ordinary objects, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
         throws dSelfException, NonLocalReturnException{

    try{
     
      switch(msg.getMessageID()){

        case PrimMsg.ADDSLOTCONNECTEDTO:        
          return _AddSlotConnectedTo((DataSO) msg.getFirstArg(), 
	                             (DataSO) msg.getSecondArg());

        case PrimMsg.ADDSLOTIFABSENTCONNECTEDTO:        
          return _AddSlotIfAbsentConnectedTo((DataSO) msg.getFirstArg(), 
	                             (DataSO) msg.getSecondArg());

        case PrimMsg.ADDSLOTS:        
          return _AddSlots((DataSO) msg.getFirstArg());

        case PrimMsg.ADDSLOTSIFABSENT:      
          return _AddSlotsIfAbsent((DataSO) msg.getFirstArg());

        case PrimMsg.DEFINE:        
          return _Define((DataSO) msg.getFirstArg());

        case PrimMsg.REMOVEALLSLOTS:
          return _RemoveAllSlots();

        case PrimMsg.REMOVESLOT:
 	  return _RemoveSlot(msg.getFirstArg());

      }

      return super.dispatchPrimMsg(msg);       
    }
    catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Adds an slot with a reference to a remote lobby to this object.
  * The first argument specifies the name of the new slot and the
  * second argument specifies a string with the URL of the remote 
  * lobby. The URL must have the following form: '//host:port/vm-name'
  * The host and port are optional and assumed to be the localhost
  * with port number 1099. The name of the remote dSelf virtual name is
  * the name, to which it was bound at the RMI-Registry. 
  *
  * @param newSlotName The name for the new slot. The argument 
  * type must be StringSO and is checked by the method
  * @param host The URL of the remote dSelf virtual machine. The argument 
  * type must be StringSO and is checked by the method
  * @return Returns this object
  */  
  protected DataSO _AddSlotConnectedTo(DataSO newSlotName, DataSO host) 
            throws dSelfException{

    checkIfStringSO("_AddSlot:ConnectedTo:", newSlotName); 
    checkIfStringSO("_AddSlot:ConnectedTo:", host); 

    String		slotName = ((StringSO)newSlotName).getString();
    String		nameOfVM = ((StringSO)host).getString();
    boolean		isParent = isParentSlot(slotName, "_AddSlot:ConnectedTo:");
    ServerInterface	vm = null;
    RemoteOrdinarySO	rmiSO = null;
    
    // Try to connect to the remote VM specified by the user and
    // generate a "RemoteOrdinarySO", that realizes the reference 
    // to the remote object
    try{
      vm = (ServerInterface) Naming.lookup(nameOfVM);
      rmiSO = new RemoteOrdinarySO(vm, vm.connectToLobby()); 
    }catch(NotBoundException e){
      throw new dSelfException(
         "A VM with this name is not bound to the registry !",
	 "_AddSlot:ConnectedTo:", "networkError");
    }catch(AccessException e){
      throw new dSelfException(
         "No permissions to access the demanded VM !",
	 "_AddSlot:ConnectedTo:", "networkError");
    }catch(RemoteException e){
      throw new dSelfException(
         "The registry of the demanded VM cannot be contacted !",
	 "_AddSlot:ConnectedTo:", "networkError");
    }catch(MalformedURLException e){
      throw new dSelfException(
         "The given URL is malformed !",
	 "_AddSlot:ConnectedTo:", "networkError");
    }

    if(isParent)
      addSlots(new SlotVector(new ParentSlot(
        slotName.substring(0, slotName.length()-1), rmiSO)));
    else
      addSlots(new SlotVector(new DataSlotImpl(slotName, rmiSO)));
       
    return this;
  }

 /**
  * Does the same like {@link #_AddSlotConnectedTo(DataSO, DataSO)}, but
  * only if no slot with the same name exist.
  *
  * @param newSlotName The name for the new slot. The argument 
  * type must be StringSO and is checked by the method
  * @param host The URL of the remote dSelf virtual machine. The argument 
  * type must be StringSO and is checked by the method
  * @return Returns this object
  */
  protected DataSO _AddSlotIfAbsentConnectedTo(DataSO newSlotName, DataSO host) 
            throws dSelfException{

    checkIfStringSO("_AddSlotIfAbsent:ConnectedTo:", newSlotName); 
    checkIfStringSO("_AddSlotIfAbsent:ConnectedTo:", host); 

    String slotName = ((StringSO)newSlotName).getString();
     
    if(getSlotContent(slotName) == null)
      return _AddSlotConnectedTo(newSlotName, host);
    else 
      return this;
  }      

 /**
  * Adds the slots of the argument object to this object. If
  * the argument object has one or more slots with names, that 
  * already exist in this object, then the old slots with the 
  * same name will be removed.
  *
  * @param arg The object with the slots, that will be added
  * @return Returns this object
  */
  protected DataSO _AddSlots(DataSO arg) throws dSelfException{
  
    addSlots(arg.getSlotVector());
    return this;
  }
  
 /**
  * Adds the slots of the argument object to this object. If
  * the argument object has one or more slots with names, that 
  * already exist in this object, then the new slots with the 
  * same name won't be added.
  *
  * @param arg The object with the slots, that will be added
  * @return Returns this object
  */
  protected DataSO _AddSlotsIfAbsent(DataSO arg) throws dSelfException{
  
    addSlotsIfAbsent(arg.getSlotVector());
    return this;
  }

 /**
  * Defines this object with the given argument.
  *
  * @param arg The object, that defines how this object shall look like
  * @return The defined object
  */
  protected DataSO _Define(DataSO arg) throws dSelfException{

    _RemoveAllSlots();
    _AddSlots(arg);
    return this;
  }
    
 /**
  * Removes all slots of this objects.
  *
  * @return Returns this object
  */ 
  protected DataSO _RemoveAllSlots(){
  
    removeAllSlots();
    return this;
  }
  
 /**
  * Removes the slot with the specified name. When no slot with
  * such a name is found an exception is thrown
  *
  * @param remSlot The name of the slot. The argument 
  * type must be StringSO and is checked by the method
  * @return Returns this object
  */ 
  protected DataSO _RemoveSlot(dSelfObject remSlot) throws dSelfException{  
  
    checkIfStringSO("_RemoveSlot:", remSlot);
    removeSlot(((StringSO)remSlot).string);
    return this;
  }  			
}
