package dSelf;

import java.rmi.*;
import java.util.Vector;
import java.net.InetAddress;
import dSelfVM;

/**
 * RemoteOrdinarySO represents an ordinary object, that is
 * located on another virtual machine. It realizes a remote reference
 * to this ordinary object and forwards all messages to it.
 */
public class RemoteOrdinarySO extends OrdinarySO{

  /** The server of the remote dSelf virtual machine. */
  protected ServerInterface remoteVM = null;

 /** 
  * The ID of the remote ordinary object, that is registered
  * at the remote server.
  */
  protected int remoteID;

  /** The name of the remote ordinary object. */
  protected String name = null;
   
 /**
  * Creates a new reference to a remote ordinary object.
  *
  * @param vm The server of the remote dSelf virtual machine
  * @param ID The ID of the remote ordinary object, that is registered
  * at the remote server 
  */ 
  public RemoteOrdinarySO(ServerInterface vm, int ID){

    remoteVM = vm;
    remoteID = ID;
  }

 /**
  * When this object is removed, it removes its corresponding reference
  * at the server of the remote dSelf virtual machine.
  */
  protected void finalize() throws RemoteException{

    remoteVM.removeRemoteReference(remoteID);
  }
  
 /**
  * Returns a vector with the contents of all parent slots.
  */
  public Vector getParentVector(){
  
    SerializedSO parents[] = null;
    
    try{
      parents = remoteVM.getParentsOf(remoteID);
    }catch(Exception e){
      handleException(e);
      return new Vector();
    }    

    Vector parVector = new Vector(parents.length);
    
    for(int i=0; i<parents.length; i++){
      if(parents[i] instanceof SerializedOrdinarySO)
        parVector.add(new RemoteOrdinarySO(remoteVM, 
	    ((SerializedOrdinarySO)parents[i]).getID()));
      else if(parents[i] instanceof SerializedObjectVectorSO)
        parVector.add(new RemoteObjectVectorSO(remoteVM, 
	    ((SerializedObjectVectorSO)parents[i]).getID())); 
      else
        parVector.add(parents[i]);
    }      

    return parVector;
  }  

 /**
  * Returns the name of this object. 
  */ 
  public String getName(){
  
    // The names of objects don't change, so we can make a local copy of it
    // without struggling with inconsistencies.
    try{
      if(name == null)
        name = remoteVM.getNameOf(remoteID)+"(r)";  
	
      return name; 
    }catch(Exception e){
      handleException(e);
      return "???";
    }    
  }  

  /** The hashcode of the remote object */
  private int remHash = 0;
  
  /** The name of the dSelfVM, where this object is located */
  private String locationVM = null;
  
  /** Returns the hashcode of the object that is refered by this reference*/
  protected int getHashCode(){

    try{
      if(remHash == 0)  
        remHash = remoteVM.getHashCodeOf(remoteID);
    
      return remHash;  
    }catch(Exception e){
      handleException(e);
      return 0;
    }    
  }

  /** Returns the name of the VM, where this object is located */
  protected String getLocation(){
  
    try{
      if(locationVM == null);
        locationVM = remoteVM.getLocationOf(remoteID);
      
      return locationVM;
    }catch(Exception e){
      handleException(e);
      return "";
    }    
  }
  
 /** 
  * Returns the content of the slot with the specified name.
  *
  * @param slotName The name of the demanded slot
  * @return The content of the found slot or null if no one was found
  */
  public dSelfObject getSlotContent(String slotName){
 
    SerializedSO result = null;

    // Get the content of a remote Slot. If the content is serializable,
    // then this object is returned, otherwise a reference to the remote
    // object has been created at the remote VM.    
    try{
      result = remoteVM.getSlotContentOf(remoteID, slotName);
    }catch(Exception e){
      handleException(e);
      return new NilSO();
    }    
    
    // The remote object is an ordinary object, so return the reference to it
    if(result instanceof SerializedOrdinarySO)
      return new RemoteOrdinarySO(remoteVM, ((SerializedOrdinarySO)result).getID());
				      
    // The remote object is a vector, so return the reference to it
    if(result instanceof SerializedObjectVectorSO)
      return new RemoteObjectVectorSO(remoteVM, 
          ((SerializedObjectVectorSO)result).getID());

    // The remote object is a assignment, so return the reference to it
    if(result instanceof SerializedAssignmentSO)
      return new RemoteAssignmentSO(remoteVM, 
          ((SerializedAssignmentSO)result).getID());

    // The remote object is a method
    if(result instanceof SerializedMethodSO)
      return ((SerializedMethodSO)result).getMethod(remoteVM);

    // The remote object is a block
    if(result instanceof SerializedBlockSO)
      return ((SerializedBlockSO)result).getBlock(remoteVM);
    
    // The remote object is a serialized primitive object, so just
    // give it back.
    return (dSelfObject) result;
  }

 /**
  * Returns the slot vector of this data object.
  */
  public SlotVector getSlotVector(){

    SerializedSlotVector remSlots = null;
     
    try{
      remSlots = remoteVM.getSlotsOf(remoteID);
    }catch(Exception e){
      handleException(e);
      return new SlotVector();
    }    
    
    return remSlots.getSlotVector(remoteVM);
  }

 /**
  * Adds some slots to this object. If the given slot vector
  * has one or more slots with names, that already exist in this 
  * object, then the old slots with the same name will be removed.
  *
  * @param slots The slots, that will be added
  */
  protected void addSlots(SlotVector slots) throws dSelfException{
  
    try{	
      remoteVM.addSlotsTo(remoteID, dSelfVM.VMName, 
              new SerializedSlotVector(slots, dSelfVM.server));
    }catch(Exception e){
      handleException(e);
    }    
  }
  
 /**
  * Adds some slots to this object. If the given slot vector
  * has one or more slots with names, that already exist in this 
  * object, then the new slots with the same name aren't added.
  *
  * @param slots The slots, that will be added
  */
  protected void addSlotsIfAbsent(SlotVector slots) throws dSelfException{
  
    try{	
      remoteVM.addSlotsIfAbsentTo(remoteID, dSelfVM.VMName, 
              new SerializedSlotVector(slots, dSelfVM.server));
    }catch(Exception e){
      handleException(e);
    }    
  }
  
 /**
  * Removes all slots of this objects.
  */ 
  protected void removeAllSlots(){
  
    try{
      remoteVM.removeAllSlotsOf(remoteID);
    }catch(Exception e){
      handleException(e);
    }    
  }

 /**
  * Removes the slot with the specified name. When no slot with
  * such name is found an exception is thrown
  *
  * @param remSlot The name of the slot
  */ 
  protected void removeSlot(String slotName) throws dSelfException{  

    try{
      remoteVM.removeSlotOf(remoteID, slotName);
    }catch(Exception e){
      handleException(e);
    }    
  }
  
 /**
  * Locks this object. The argument shall be the owner of the lock
  *
  * @param so The owner of the lock
  */
  protected synchronized void lock(dSelfObject so){

    try{
      remoteVM.lockObject(remoteID, so.hashCode());
    }catch(Exception e){
      handleException(e);
    }    
  }
  
 /**
  * Unlocks this object. The argument is the owner of the lock
  *
  * @param so The owner of the lock
  */
  protected void unlock(dSelfObject so) throws dSelfException{

    try{
      remoteVM.unlockObject(remoteID, so.hashCode());
    }catch(Exception e){
      handleException(e);
    }    
  }

}
