package dSelf;

import java.io.Serializable;  

/**
 * SerializedSlotVector represents the serialized version of a
 * slot vector. 
 */
public class SerializedSlotVector implements Serializable{

  /** An array with the serialized contents of the slot vector. */
  protected SerializedSO[] remoteSOs;

  /** An array with all slotnames. */
  protected String[] slotNames;

  /** An array with the specifications of all slots. */
  protected int[] slotTypes;
  
  private final static int DATAOBJECT = 1;
  private final static int ORDINARYOBJECT = 2;
  private final static int OBJECTVECTOR = 4;
  private final static int PARENTOBJECT = 8;
  private final static int ASSIGNABLEOBJECT = 16;
  private final static int ARGUMENTOBJECT = 32;
  private final static int METHODOBJECT = 64;
  private final static int BLOCKOBJECT = 128;
   
 /**
  * Creates a new serialized slot vector based on the given slot vector.
  *
  * @param slotVec The slot vector, that shall be serialized
  * @param server The server, that will send the slot vector
  */    
  public SerializedSlotVector(SlotVector slotVec, Server server){
  
    remoteSOs = new SerializedSO[slotVec.size()];
    slotNames = new String[slotVec.size()];
    slotTypes = new int[slotVec.size()];
    
    for(int i=0; i<slotVec.size(); i++){
      Slot slot = slotVec.getSlotAt(i);
      dSelfObject obj = slot.getContent();
	
      if(obj instanceof DataOrMethodSO){
        slotNames[i] = slot.getName();

        if(obj instanceof DataSO){
 	  slotTypes[i] |= DATAOBJECT;    

	  if(slotVec.isAssignable(i))
	    slotTypes[i] |= ASSIGNABLEOBJECT; 

   	  if(slot instanceof ParentSlot)
	    slotTypes[i] |= PARENTOBJECT;

   	  if(slot instanceof ArgumentSlot)
	    slotTypes[i] |= ARGUMENTOBJECT;

          if(obj instanceof OrdinarySO){
	    slotTypes[i] |= ORDINARYOBJECT; 
            remoteSOs[i] = new SerializedOrdinarySO(
	                 server.addRemoteReference((RemoteReferenceSO)obj)); 
          }else if(obj instanceof ObjectVectorSO){
	    slotTypes[i] |= OBJECTVECTOR; 
            remoteSOs[i] = new SerializedObjectVectorSO(
	                 server.addRemoteReference((RemoteReferenceSO)obj)); 
          }else if(obj instanceof BlockSO){
	    slotTypes[i] |= BLOCKOBJECT; 
            remoteSOs[i] = new SerializedBlockSO((BlockSO)obj, server); 
          }else{
            remoteSOs[i] = (SerializedSO)obj; 
	  }  
	}else{
 	  slotTypes[i] |= METHODOBJECT;    
          remoteSOs[i] = new SerializedMethodSO((MethodSO)obj, server);
	}
      }
    }
  }

 /**
  * Returns the slot vector, that was serialized.
  *
  * @param remoteVM The server of the dSelf VM, that received this 
  * slot vector.
  */   
  public SlotVector getSlotVector(ServerInterface remoteVM){
  
    SlotVector slotVec = new SlotVector();
    slotVec.enableGenesInfo(false);
    boolean isDataObject, isOrdinaryObject, isObjectVector;
    boolean isParentObject, isAssignableObject, isArgumentObject;
    boolean isBlockObject;
    
    for(int i=0; i<remoteSOs.length; i++){
      isDataObject = (slotTypes[i] & DATAOBJECT) != 0;
      isOrdinaryObject = (slotTypes[i] & ORDINARYOBJECT) != 0;
      isObjectVector = (slotTypes[i] & OBJECTVECTOR) != 0;
      isParentObject = (slotTypes[i] & PARENTOBJECT) != 0;
      isArgumentObject = (slotTypes[i] & ARGUMENTOBJECT) != 0;
      isAssignableObject = (slotTypes[i] & ASSIGNABLEOBJECT) != 0;
      isBlockObject =  (slotTypes[i] & BLOCKOBJECT) != 0;
      
      if(remoteSOs[i] != null){
        if(isDataObject){

          DataOrMethodSlot slot = null;
	  
	  if(isParentObject)
	    slot = new ParentSlot(slotNames[i]);
	  else if(isArgumentObject)
	    slot = new ArgumentSlot(slotNames[i]);
	  else 
	    slot = new DataSlotImpl(slotNames[i]);
	  
	  if(isOrdinaryObject)
	    ((DataSlot)slot).setContent(new RemoteOrdinarySO(remoteVM, 
	        ((SerializedOrdinarySO)remoteSOs[i]).getID()));  
	  else if(isObjectVector)
	    ((DataSlot)slot).setContent(new RemoteObjectVectorSO(remoteVM, 
	        ((SerializedObjectVectorSO)remoteSOs[i]).getID()));  
	  else if(isBlockObject)
            ((DataSlot)slot).setContent(
	        ((SerializedBlockSO)remoteSOs[i]).getBlock(remoteVM));
          else 	  
	    ((DataSlot)slot).setContent((PrimitiveSO)remoteSOs[i]);
	    
	  if(isAssignableObject)
	    slotVec.addAssignableDataSlot((DataSlot)slot);
	  else
	    slotVec.add(slot);    		      
	}else{
	  slotVec.add(new MethodSlotImpl(slotNames[i], 
	      ((SerializedMethodSO)remoteSOs[i]).getMethod(remoteVM)));
	}
      }
    }
    
    return slotVec;
  }
}

/* maybe Java writeReplace() and readResolve() can be used instead? */
