package dSelf;
 
import java.io.*;
import java.util.Vector;
import dSelfVM;

/**
 * The class ShortSO repesents the short numbers of dSelf.
 * ShortSO is a primitive dSelf object and so it has only one
 * parent slot called "parent", that is shared by all shorts.
 * The object, that "parent" refers to has initialy only one
 * parent slot, also called "parent" that refers to the lobby.
 * This object is dedicated to include the user-defined methods,
 * (e.g. +, -, * ...) that are shared by all shorts.
 */
public class ShortSO extends SerializedPrimitiveSO{
 
  /** The value of the short object */
  public short value = 0;

 /** The shared parent object for all shorts */
  protected static LocalOrdinarySO parent = 
    new LocalOrdinarySO(new ParentSlot("parent", dSelfVM.lobbySO));

 /**
  * The shared slots for all shorts. It consists of only one slot
  * called "parent"
  */
  protected static SlotVector slotVector = 
    new SlotVector(new ParentSlot("parent",parent));   
    
 /**
  * The shared parent slots for all shorts. It consists of only one slot
  * called "parent"
  */
  protected static Vector parVector = new Vector();   
  
  static{
    parVector.add(parent);   
  }
  
 /**
  * Creates new dSelf short with the given Java short value
  */  
  public ShortSO(short shortValue){

    value = shortValue;
  }
    
 /**
  * Creates new dSelf short with the given Java Short value
  */  
  public ShortSO(Short shortValue){

    value = shortValue.shortValue();
  }

 /**
  * Returns the slots of the shorts. I.e, it's only one slot
  * called "parent".
  */   
   public SlotVector getSlotVector(){
    
    return slotVector; 
  }
  
 /**
  * Returns the shared parent slot for all dSelf shorts.
  */  
  protected LocalOrdinarySO getParent(){
  
    return parent;
  }
    
 /**
  * Returns the parent slots of the shorts. I.e, it's only one slot
  * called "parent".
  */   
  public Vector getParentVector(){
  
    return parVector;
  }   

 /**
  * Returns the value of this short in the form "short(...)".
  */
  public String getName(){
  
    return "short("+value+")";
  }

 /**
  * The dispatcher for dSelf shorts, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
         throws dSelfException, NonLocalReturnException{

    try{
    
      switch(msg.getMessageID()){

        case PrimMsg.EQ:                     
	  return _Eq(msg.getFirstArg());

        case PrimMsg.SHORTADD:                     
	  return _ShortAdd(msg.getFirstArg());
	
        case PrimMsg.SHORTAND:                     
	  return _ShortAnd(msg.getFirstArg());
	
        case PrimMsg.SHORTARITHMETICSHIFTRIGHT:                     
	  return _ShortArithmeticShiftRight(msg.getFirstArg());
	
        case PrimMsg.SHORTASDOUBLE:                     
	  return _ShortAsDouble();
	
        case PrimMsg.SHORTASFLOAT:                     
	  return _ShortAsFloat();

        case PrimMsg.SHORTASINT:                     
	  return _ShortAsInt();

        case PrimMsg.SHORTASLONG:                     
	  return _ShortAsLong();

        case PrimMsg.SHORTDIV:                     
	  return _ShortDiv(msg.getFirstArg());
	
        case PrimMsg.SHORTEQ:                     
	  return _ShortEQ(msg.getFirstArg());
	
        case PrimMsg.SHORTGE:                     
	  return _ShortGE(msg.getFirstArg());
	
        case PrimMsg.SHORTGT:                     
	  return _ShortGT(msg.getFirstArg());
	
        case PrimMsg.SHORTLE:                     
	  return _ShortLE(msg.getFirstArg());
	
        case PrimMsg.SHORTLT:                     
	  return _ShortLT(msg.getFirstArg());
	
        case PrimMsg.SHORTLOGICALSHIFTLEFT:                     
	  return _ShortLogicalShiftLeft(msg.getFirstArg());
	
        case PrimMsg.SHORTLOGICALSHIFTRIGHT:                     
	  return _ShortLogicalShiftRight(msg.getFirstArg());
	
        case PrimMsg.SHORTMOD:                     
	  return _ShortMod(msg.getFirstArg());
	
        case PrimMsg.SHORTMUL:                     
	  return _ShortMul(msg.getFirstArg());
	
        case PrimMsg.SHORTNE:                     
	  return _ShortNE(msg.getFirstArg());
	
        case PrimMsg.SHORTOR:                     
	  return _ShortOr(msg.getFirstArg());
	
        case PrimMsg.SHORTSUB:                     
	  return _ShortSub(msg.getFirstArg());
	
        case PrimMsg.SHORTXOR:                     
	  return _ShortXor(msg.getFirstArg());
      }

      return super.dispatchPrimMsg(msg);       
    }
    catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Checks, if the argument is equal to the value of this short object.
  *
  * @param arg The argument must be of type ShortSO
  */
  protected BooleanSO _Eq(dSelfObject so) throws dSelfException{

    checkIfShortSO("_Eq:", so);
    return BooleanSO.asBooleanSO(((ShortSO)so).value == value);
  }  

 /**
  * Adds another dSelf short to this short
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected ShortSO _ShortAdd(dSelfObject so) throws dSelfException{

    checkIfShortSO("_ShortAdd:", so);
    return new ShortSO((short)(value + ((ShortSO)so).value)); 
  }  
 
 /**
  * Returns a bitwise AND of this short and the argument
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */ 
  protected ShortSO _ShortAnd(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortAnd:", so);
    return new ShortSO((short)(value & ((ShortSO)so).value)); 
  }  

 /**
  * Arithmetic right shift of this short by the numbers of bits indicated
  * by the argument. The sign bit is preserved
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected ShortSO _ShortArithmeticShiftRight(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortArithmeticShiftRight:", so);
    return new ShortSO((short)(value >> ((ShortSO)so).value));
  }    
  
 /**
  * Converts this dSelf short to a dSelf double.  
  */
  protected DoubleSO _ShortAsDouble(){
  
    return new DoubleSO((double) value);  
  }  

 /**
  * Converts this dSelf short to a dSelf float.  
  */
  protected FloatSO _ShortAsFloat(){
  
    return new FloatSO((float) value);  
  }  

 /**
  * Converts this dSelf short to a dSelf long.  
  */
  protected LongSO _ShortAsLong(){
  
    return new LongSO((long) value);  
  }  

 /**
  * Converts this dSelf short to a dSelf integer.  
  */
  protected IntegerSO _ShortAsInt(){
  
    return new IntegerSO((int) value);  
  }  

 /**
  * Divides this short through another short
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected ShortSO _ShortDiv(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortDiv:", so);
    try{  
      return new ShortSO((short)(value / ((ShortSO)so).value)); 
    }catch(ArithmeticException e){      
      throw new dSelfException(" Division by zero -> "+value+
	  " _ShortDiv: 0 !", "_ShortDiv", "divisionByZeroError");
    }  
  }  
  
 /**
  * Checks, if the argument is equal to the value of this short object.
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected BooleanSO _ShortEQ(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortEQ:", so);
    return BooleanSO.asBooleanSO(value == ((ShortSO)so).value);
  }
  
 /**
  * Checks, if this short is greater or equal than its argument.
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected BooleanSO _ShortGE(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortGE:", so);
    return BooleanSO.asBooleanSO(value >= ((ShortSO)so).value);
  }
      
 /**
  * Checks, if this short is greater than its argument.
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected BooleanSO _ShortGT(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortGT:", so);
    return BooleanSO.asBooleanSO(value > ((ShortSO)so).value);
  }
  
 /**
  * Checks, if this short is less or equal than its argument.
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected BooleanSO _ShortLE(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortLE:", so);
    return BooleanSO.asBooleanSO(value <= ((ShortSO)so).value);
  }
  
 /**
  * Checks, if this short is less than its argument.
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected BooleanSO _ShortLT(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortLT:", so);
    return BooleanSO.asBooleanSO(value < ((ShortSO)so).value);
  }
 
 /**
  * Logical left shift of this short by the numbers of bits indicated
  * by the argument. 
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected ShortSO _ShortLogicalShiftLeft(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortLogicalShiftLeft:", so);
    return new ShortSO((short)(value << ((ShortSO)so).value));
  }    

 /**
  * Logical right shift of this short by the numbers of bits indicated
  * by the argument. 0 is shifted into the most significant bit.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */  
  protected ShortSO _ShortLogicalShiftRight(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortLogicalShiftRight:", so);
    return new ShortSO((short)(value >>> ((ShortSO)so).value));
  }    
  
 /**
  * Returns this short modulo argument. 
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected ShortSO _ShortMod(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortMod:", so);
    try{  
      return new ShortSO((short)(value % ((ShortSO)so).value));
    }
    catch(ArithmeticException e){
      if(((ShortSO)so).value == 0)
        throw new dSelfException(" Division by zero -> "+value+
	       " _ShortMod: 0 !", "ShortMod:", "divisionByZeroError");
    }  
    
    return null;
  }  
  
 /**
  * Returns the product of this short and the argument. 
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected ShortSO _ShortMul(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortMul:", so);
    return new ShortSO((short)(value * ((ShortSO)so).value)); 
  }  
  
 /**
  * Checks if this short is inequal to its receiver
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */ 
  protected BooleanSO _ShortNE(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortNE:", so);
    return BooleanSO.asBooleanSO(value != ((ShortSO)so).value);
  }
  
 /**
  * Returns a bitwise OR of this short and the argument
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */ 
  protected ShortSO _ShortOr(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortOr:", so);
    return new ShortSO((short)(value | ((ShortSO)so).value));
  }  
  
 /**
  * Returns this short minus the argument.
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */
  protected ShortSO _ShortSub(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortSub:", so);
    return new ShortSO((short)(value - ((ShortSO)so).value)); 
  }  
  
 /**
  * Returns a bitwise XOR of this short and the argument
  *
  * @param arg The argument type must be ShortSO and is checked by
  * the method
  */ 
  protected ShortSO _ShortXor(dSelfObject so) throws dSelfException{
  
    checkIfShortSO("_ShortXor:", so);
    return new ShortSO((short)(value ^ ((ShortSO)so).value));  
  }  
}   
