package dSelf;

import java.util.Vector;
import java.lang.ref.WeakReference;
import java.lang.ref.ReferenceQueue;
import dSelfVM;

/**
 * SlotVector consist of slots and provides methods for adding and
 * accessing them.
 */
public class SlotVector {

  /** When index == -1 is returned, then no matching slot was found. */
  public static final int NO_SLOT_FOUND = -1;

  /** intern representation of slots */
  private Vector theSlots = new Vector();

  /** if there is at least one method slot, it will be assign a uniq gene */
  private Long ownGene = null;

  /** calculatd genes inherited from parents */
  private Genes inheritedGenes = new Genes();

  /** combined genes from ownGene and inheritedGenes */
  private Genes genes = new Genes();

  /** children which must be updated when genes changes */
  private Vector children = new Vector();
  /** children will contain weak reference associated with this queue */
  private ReferenceQueue lostChildren = new ReferenceQueue();

  /** if this SlotVector carries genes */
  private boolean haveGenes = true;
  
 /**
  * Creates a new, empty slot vector
  */  
  public SlotVector() {}

 /**
  * Creats a new slot vector with the given slot as its content.
  */
  public SlotVector(Slot slot){

    add(slot);  
  }

  public void enableGenesInfo(boolean enable) {
    haveGenes = enable;
  }

  /* not use other Vector methods, except following add/remove, etc. */

  public boolean add(Object o) {

    boolean result = theSlots.add(o);
    if (haveGenes)
      updateGenes((Slot)o, true);
    
    return result;
  }

  public Object remove(int index) {
    Object result = theSlots.remove(index);
    if (haveGenes)
      updateGenes((Slot) result, false);

    return result;
  }

  public int size() {
    return theSlots.size();
  }
  
  public String toString() {
    return theSlots.toString();
  }

  /* genes manipulation */

  private void updateGenes(Slot s, boolean add) {

    if (! (s instanceof ParentSlot)) {
      if (add) {
	if (Globals.debug_lookupCache)
	  dSelfVM.printMessage("add MethodSlot: "+s.getName()+" +> "+printSlots());
	ownGene = dSelfVM.getNextGeneCode();
      }else {
	if (Globals.debug_lookupCache)
	  dSelfVM.printMessage("remove MethodSlot:"+s.getName()+" -> "+printSlots());

	boolean hasOwnSlots = false;
	for (int i=0; i<this.size(); i++)
	  if (! (getSlotAt(i) instanceof ParentSlot)) {
	    hasOwnSlots = true;
	    break;
	  }

	if (hasOwnSlots) {
	  ownGene = dSelfVM.getNextGeneCode();
	}else
	  ownGene = null;
      }
      combineGenes();
      propagateGenes();

    }else if (s instanceof ParentSlot) {
      if (updateInheritedGenes()) {
	combineGenes();
	propagateGenes();
      }
      
      SlotVector p = ((DataOrMethodSO)s.getContent()).getSlotVector();
      if (add)
	p.registerChild(this);
      else
	p.unregisterChild(this);
    }
  }

  private boolean updateInheritedGenes() {
    Vector parents = getParentVector();
    Genes tmp = new Genes();
    Genes tmp2 = tmp;

    if (Globals.debug_lookupCache)
      dSelfVM.printMessage("current inherited genes: "+inheritedGenes);

    for (int i=0; i<parents.size(); i++)
      tmp.addAll(((DataOrMethodSO)parents.get(i)).getSlotVector().getGenes());

    if (ownGene != null) {
      tmp2 = new Genes(tmp);
      tmp2.add(ownGene);
    }

    if (tmp2.equals(this.genes))
      return false;		// no need to update
      
    inheritedGenes = tmp;
    if (Globals.debug_lookupCache)
      dSelfVM.printMessage("new inherited genes: "+inheritedGenes);

    return true;
  }

  private void propagateGenes() {
    clearLostChildren();
    for (int i=0; i<children.size(); i++) {
      SlotVector child = (SlotVector)((WeakReference)children.get(i)).get();

      if (child != null && child.updateInheritedGenes()) {
	child.combineGenes();
	child.propagateGenes();
      }
    }      
  }

  private void registerChild(SlotVector s) {
    clearLostChildren();
    children.add(new WeakReference(s, lostChildren));

    if (Globals.debug_lookupCache)
      dSelfVM.printMessage("registerChild +> "+children);
  }

  private void unregisterChild(SlotVector s) {
    clearLostChildren();
    for (int i=0; i<children.size(); i++) {
      WeakReference ref = (WeakReference)children.get(i);
      SlotVector c = (SlotVector)ref.get();

      if (c != null && c.equals(s)) {
	children.remove(ref);
	break;
      }
    }

    if (Globals.debug_lookupCache)
      dSelfVM.printMessage("unregisterChild -> "+children);
  }

  private void clearLostChildren() {
    WeakReference ref;
    while ((ref = (WeakReference)lostChildren.poll()) != null) {
      children.remove(ref);

      if (Globals.debug_lookupCache)
	dSelfVM.printMessage("clearLostChildren: "+ref);
    }
  }

  private void combineGenes() {
    Genes tmp = new Genes();

    if (ownGene != null)
      tmp.add(ownGene);

    tmp.addAll(inheritedGenes);
    this.genes = tmp;

    if (Globals.debug_lookupCache)
      dSelfVM.printMessage("combined genes => "+this.genes);
  }

  protected Genes getGenes() { return this.genes; }

 /**
  * Returns a clone of this slot vector.
  */
  public Object clone(){
  
    SlotVector result = new SlotVector();
    Slot slot = null;
    DataOrMethodSlot slotClone = null;

    result.theSlots = (Vector) this.theSlots.clone();
    
    for(int i=0; i<size(); i++){
      slot = getSlotAt(i);
      if(slot instanceof DataOrMethodSlot){
        slotClone = (DataOrMethodSlot)((DataOrMethodSlot)slot).clone();
        if(isAssignable(i)) {
          result.setSlotAt(i, slotClone);
	  result.setSlotAt
	    (++i, new AssignmentSlot(slotClone.getName()+":", (DataSlot)slotClone));
	}
        else 
          result.setSlotAt(i, slotClone);
      }	    
    }

    result.ownGene = this.ownGene;
    result.inheritedGenes = this.inheritedGenes;
    result.genes = this.genes;
//     result.children = new Vector(this.children); // ???
    
    if (Globals.debug_lookupCache || Globals.debug_searchPath)
      dSelfVM.printMessage("SlotVector is cloned: "+this);
    return result;
  }
  
 /**
  * Returns the index of the slot, whose name matches with the given
  * name.
  *
  * @param slotName The name of the demanded slot
  * @return The index of the slot. If no one was found {@link #NO_SLOT_FOUND}
  * is returned.
  */ 
  public int getSlotIndex(String slotName){

    for(int i=0; i<size(); i++)
      if(getSlotAt(i).getName().equals(slotName))
        return i;
	
    return NO_SLOT_FOUND;	
  }

 /**
  * Returns the slot, whose name matches with the given name.
  *
  * @param slotName The name of the demanded slot
  * @return The slot, that was found. If no one was found, then null
  * is returned.
  */ 
  public Slot getSlot(String slotName){

    int index = getSlotIndex(slotName);
    
    return  index == NO_SLOT_FOUND ? null: getSlotAt(index);
  }

 /**
  * Returns the slot at the given index. 
  *
  * @param index An index of this vector
  * @return The slot, that was found. If the index isn't valid, then null
  * is returned.
  */     
  public Slot getSlotAt(int index){

    return theSlots.get(index) == null ? null : (Slot) theSlots.get(index);
  }  

  public void setSlotAt(int index, Slot s) {
    theSlots.set(index, s);
  }

  public void clearAllSlots() {
    theSlots = new Vector();
    ownGene = null;
    inheritedGenes = new Genes();
    genes = new Genes();
    children = new Vector();
    lostChildren = new ReferenceQueue();
  }
  
 /**
  * Returns the content of the slot, whose name matches with the given 
  * name.
  *
  * @param slotName The name of the demanded slot
  * @return The content of the slot, that was found. If no one was found, 
  * then null is returned.
  */ 
  public dSelfObject getSlotContent(String slotName){

    int index = getSlotIndex(slotName);
    
    return index == NO_SLOT_FOUND ? null : getSlotContent(index);
  }

 /**
  * Returns the conmtent of the slot at the given index. 
  *
  * @param index An index of this vector
  * @return The content of the slot, that was found. If the index isn't 
  * valid, then null is returned.
  */     
  public dSelfObject getSlotContent(int slotNr){
 
    Slot slot = getSlotAt(slotNr);

    return slot == null ? null : slot.getContent();
  }

 /**
  * Returns a slot vector with all parent slots of this slot vector.
  */
  public Vector getParentVector(){
 
    Vector result = new Vector();
    
    for(int i=0; i<size(); i++)
      if(getSlotAt(i) instanceof ParentSlot)
         result.add(getSlotContent(i));
       
    return result;
  }

 /**
  * Adds a new slot to this slot vector. If another slot with the same
  * name is found, then it is replaced by this one. If the old one was
  * assignable, then the new slot will also be assignable.
  *
  * @param newSlot The slot, that shall be added.
  */
  public void addSlot(Slot newSlot){
  
    int index = getSlotIndex(newSlot.getName());
    boolean wasAssignable = isAssignable(index);
    
    if(index == NO_SLOT_FOUND){
      add(newSlot);
    }else{
      if(wasAssignable){
        if(newSlot instanceof DataSlot){
          removeSlot(index);
          add(newSlot);
          add(new AssignmentSlot(newSlot.getName()+":", (DataSlot)newSlot));
	}else{  
//           remove(index+1);
	  removeSlot(index);
          add(newSlot);
        }	  
      }else{
        removeSlot(index);
        add(newSlot);
      }	
    } 
  }
    
 /**
  * Adds a new slot to this slot vector. If another slot with the same
  * name is found, then nothing is done.
  *
  * @param newSlot The slot, that shall be added.
  */
  public void addSlotIfAbsent(Slot newSlot){
  
    if(getSlotIndex(newSlot.getName()) == NO_SLOT_FOUND)
      addSlot(newSlot);
  }

 /**
  * Adds the given data slot to this slot vector and creates a 
  * corresponding assignment slot with a trailing colon. If another
  * data slot with the same name exists, it is replaced by the new
  * one.
  * 
  * @param slot The new data slot, that shall be added
  */ 
  protected void addAssignableDataSlot(DataSlot slot){
  
    int index = getSlotIndex(slot.getName());

    if(index != NO_SLOT_FOUND)
      removeSlot(index);

    add(slot);
    add(new AssignmentSlot(slot.getName()+":", slot));
  }  

 /**
  * Adds the given data slot to this slot vector and creates a 
  * corresponding assignment slot with a trailing colon. If another
  * data slot with the same name exists, the new data slot isn't 
  * added. If the already existing slot wasn't assignable, it will
  * become assignable.
  *
  * @param slot The new data slot, that shall be added
  */
  protected void addAssignableDataSlotIfAbsent(DataSlot slot){
  
    int index = getSlotIndex(slot.getName());

    if(index == NO_SLOT_FOUND){
      addAssignableDataSlot(slot);
    }else{   
//       if(!isAssignable(index)){
//          addAssignableDataSlot((DataSlot)getSlotAt(index));
      if(getSlotAt(index) instanceof DataSlot && !isAssignable(index)) {
         addAssignableDataSlot(slot);
      }	 
    } 
  }  

 /**
  * Adds all slots of the given slot vector to this slot vector. All
  * slots, whose names match with the new ones are replaced.
  *
  * @param newSlots A slot vector with the new slots
  */ 
  public void addSlotVector(SlotVector newSlots){

    Slot newSlot = null;

    for(int i=0; i<newSlots.size(); i++){
      newSlot = newSlots.getSlotAt(i);

      if(newSlots.isAssignable(i)){
        addAssignableDataSlot((DataSlot)newSlot);
	i++;
      }else{
        addSlot(newSlot);
      }	
    }  
  }

 /**
  * Adds all slots of the given slot vector to this slot vector. All
  * slots, whose names match with the new ones aren't replaced.
  *
  * @param newSlots A slot vector with the new slots
  */ 
  public void addSlotVectorIfAbsent(SlotVector newSlots){
  
    Slot newSlot = null;

    for(int i=0; i<newSlots.size(); i++){
      newSlot = newSlots.getSlotAt(i);

      if(newSlots.isAssignable(i)){
        addAssignableDataSlotIfAbsent((DataSlot)newSlot);
	i++;
      }else{
        addSlotIfAbsent(newSlot);
      }	
    }  
  }

 /**
  * Removes the slot at the specified position in this slot vector.
  *
  * @param slotIndex The index of the slot
  * @return False, when the index was out of range of this vector
  */
  protected boolean removeSlot(int slotIndex){

    if(slotIndex != NO_SLOT_FOUND){
      if(isAssignable(slotIndex))
        remove(slotIndex);

      remove(slotIndex);
      return true;
    }	
    return false;
  }
  
  public boolean removeSlot(String slotName){

    return removeSlot(getSlotIndex(slotName));
  }  
  
 /**
  * Returns a string that consists of all slots and their values.
  */           
  public String printSlots(){

    String s = "";
    Slot slot;
    
    for(int i=0; i<size(); i++){
      slot = getSlotAt(i);
      if(!(slot instanceof AssignmentSlot))
        s += slot.getName() +
            (slot instanceof ParentSlot ? "*" : "") + 
	    (isAssignable(i) ? " <- " : " = ") +
            ((DataOrMethodSlot)slot).getContent().getName() + ". ";
    }
                                              
    return "(| "+s+" |)";	    
  }

 /**
  * Returns an array of StringSO with all slot names.
  */  
  
  public StringSO[] getSlotNames(){
  
    StringSO[] names = new StringSO[size()];
    
    for(int i=0; i<size(); i++)
      names[i] = new StringSO(getSlotAt(i).getName());
    
    return names;    
  }
  
 /**
  * Returns a string with all slots and their annotations.
  */
  public String describeSlots(){
  
    Slot slot;
    String s = "";
    
    for(int i=0; i<size(); i++){
      slot = getSlotAt(i);
      s += " Slot \""+slot.getName()+"\": \t"+slot.getAnnotation()+"\n";
    }
                                              
    return s;	    
  }

 /**
  * Checks if the slot at the specified position is assignable or not.
  */  
  public boolean isAssignable(int pos){
    
    // Assignable slots always occur directly behind their corresponding
    // data slots.
    try{ 
      if((getSlotAt(pos) instanceof DataSlot) && 
          (getSlotAt(pos+1) instanceof AssignmentSlot))
	return true;
    }catch(ArrayIndexOutOfBoundsException e){}

    return false;
  }  	 
  
 /**
  * Adds the given annotation to all slots of this slot vector.
  */
  public SlotVector setAnnotations(String a){
  
    for(int i=0; i<size(); i++)
      getSlotAt(i).setAnnotation(a);
      
    return this;  
  }
  
 /**
  * Initializes the argument slots of this slot vector with the values
  * of the vector. This method is called, when this slot vector is
  * part of a dSelf method or block. In case of a method then the string 
  * specifies the name of the method, that is needed, when an error
  * message is printed, in case of to few or many arguments. When it is
  * block method. then an error is only thrown, when to few 
  * argument slots were declared.
  *
  * @param args A vector with the values for the argument slots
  * @param methodName The name of the method, which is the owner of
  * this slot vector.
  */ 
  public void initArgs(Vector args, String methodName) throws dSelfException{

    int numArgs = 0;
    ArgumentSlot slot = null;

    try{
      for(int i=0; i<size(); i++){ 
        if(getSlotAt(i) instanceof ArgumentSlot){
	  slot = (ArgumentSlot)getSlotAt(i);
	  slot.setContent((DataSO)args.get(numArgs));
          numArgs++;
        }
      }  
    }catch(ArrayIndexOutOfBoundsException e){  
      throw new dSelfException("Wrong number of argument slots in \""+methodName+"\" !\n"+ 
	                          "Too many ("+numArgs+") were declared.");
    }
    
    // If too many arguments were send and the method isn't embedded in a 
    // block, then give an alert      
    if(numArgs < args.size() && ! Message.isBlockValueSelector(methodName))
      throw new dSelfException("Wrong number of argument slots in method \""+
               methodName+"\" !\nToo few ("+numArgs+")  were declared.");
  }
}
