package dSelf;

import java.io.*;
import java.util.Vector;
import dSelfVM;

/**
 * The class StringSO repesents the strings of dSelf.
 * StringSO is a primitive dSelf object and so it has only one
 * parent slot called "parent", that is shared by all strings.
 * The object, that "parent" refers to has initialy only one
 * parent slot, also called "parent" that refers to the lobby.
 * This object is dedicated to include the user-defined methods,
 * (e.g. +, -, * ...) that are shared by all strings.
 */
public class StringSO extends SerializedPrimitiveSO{

  /** The value of the string object */
  protected String string = null;

 /** The shared parent object for all strings */
  protected static LocalOrdinarySO parent = 
    new LocalOrdinarySO(new ParentSlot("parent", dSelfVM.lobbySO));

 /**
  * The shared slots for all strings. It consists of only one slot
  * called "parent"
  */
  protected static SlotVector slotVector = 
    new SlotVector(new ParentSlot("parent",parent));   
    
 /**
  * The shared parent slots for all strings. It consists of only one slot
  * called "parent"
  */
  protected static Vector parVector = new Vector();   
  
  static{
    parVector.add(parent);   
  }
  
 /**
  * Creates new dSelf string with the given Java string value
  */  
  public StringSO(String str){
  
    string = str;
  }

 /**
  * Returns the slots of the strings. I.e, it's only one slot
  * called "parent".
  */   
  public SlotVector getSlotVector(){
    
    return slotVector; 
  }
  
 /**
  * Returns the shared parent slot for all dSelf strings.
  */  
  protected LocalOrdinarySO getParent(){
  
    return parent;
  }
    
 /**
  * Returns the parent slots of the strings. I.e, it's only one slot
  * called "parent".
  */   
  public Vector getParentVector(){
  
    return parVector;
  }   

/**
 * Returns a clone of this object.
 */ 
 public Object clone(){
  
    return new StringSO(new String(string));
  }
 
 /**
  * Returns the current value of ths string.
  */       
  public String getString(){
  
    return string;
  }

 /**
  * Returns the value of this string in the form "string(...)".
  */
  public String getName(){
  
    return "string("+string+")";
  }

 /**
  * The dispatcher for dSelf strings, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
         throws dSelfException, NonLocalReturnException{

    try{
     
      switch(msg.getMessageID()){

        case PrimMsg.BITSIZE:           
          return _BitSize();

        case PrimMsg.EQ:                     
	  return _Eq(msg.getFirstArg());

        case PrimMsg.RUNSCRIPT:                     
	  return _RunScript();  

        case PrimMsg.STRINGASBYTEVECTOR:           
          return _StringAsByteVector();

        case PrimMsg.STRINGAT:           
	  return _StringAt(msg.getFirstArg());
        
        case PrimMsg.STRINGCONCATENATE:           
	  return _StringConcatenate(msg.getFirstArg());

        case PrimMsg.STRINGGETSUBFROM:           
	  return _StringGetSubFrom(msg.getFirstArg());

        case PrimMsg.STRINGGETSUBFROMTO:           
	  return _StringGetSubFromTo(msg.getFirstArg(), msg.getSecondArg());
	
        case PrimMsg.STRINGPRINT:                     
	  return _StringPrint();

        case PrimMsg.STRINGSIZE:           
          return _StringSize();
      }

      return super.dispatchPrimMsg(msg);       
    }
    catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Returns an integer, the number of bits in the representation of 
  * that type. The following strings are valid receivers: 'integer',
  * 'long', 'short', 'float' and 'double'.
  *
  * @return The bit size of the specified type
  */
  protected IntegerSO _BitSize() throws dSelfException{
  
    if(string.equals("short"))
      return new IntegerSO(16);
    if(string.equals("integer") || string.equals("float"))
      return new IntegerSO(32);
    else
      if(string.equals("long") || string.equals("double"))
        return new IntegerSO(64);
    
    throw new dSelfException("Receiver of primitive \"_BitSize:\""+
         " is not a type of dSelf !", "_BitSize", "primitiveFailedError");   
  }

 /**
  * Checks, if the argument is equal to the value of this string object.
  *
  * @param arg The argument must be of type StringSO
  */
  protected BooleanSO _Eq(dSelfObject arg) throws dSelfException{

    checkIfStringSO("_Eq:", arg); 
    return BooleanSO.asBooleanSO(((StringSO)arg).string.equals(string));
  }  

 /**
  * Reads in a dSelf script, which name is specified in this string. 
  * If a precompiled Java class with the same and newer date exists,
  * it will be linked dynamically to running environment. Otherwise
  * the dSelf script is loaded and its expressions are evaluated.
  * 
  * @return The value of the last evaluated expression in the script
  */
  protected DataSO _RunScript() 
            throws dSelfException, NonLocalReturnException{
  
    try{
      return dSelfVM.loadScriptDirPath(string).eval(dSelfVM.lobbySO);
    }
    catch(dSelfException e){  
      throw new dSelfException(e.getMessage(), "_RunScript", e.getError());
    }
  }  

 /**
  * Converts this string into a byte vector. It might happen, that 
  * informations get lost, because strings are unicode (16 bit) and
  * bytes have only 8 bit.
  *
  * @return The string as a byte vector 
  */
  protected ByteVectorSO _StringAsByteVector(){
  
    return new ByteVectorSO(string.getBytes());
  }
  
 /**
  * Returns the value of the string at the specified position. If the
  * position is to low or high, a badIndexError will return.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  * @return The value of the character at the specified position
  */ 
  protected IntegerSO _StringAt(dSelfObject arg) throws dSelfException{

    checkIfIntegerSO("_StringAt:", arg); 

    try{
      return new IntegerSO((int) string.charAt(((IntegerSO)arg).value));
    }
    catch(StringIndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive \"_StringAt:\""+
            " is out of bounds !", "_StringAt:", "badIndexError"); 
    }  
  }
 
 /**
  * Concatenates this string and the given argument, which is also a
  * string.
  *
  * @param arg The argument type must be StringSO and is checked by
  * the method
  * @return The concatenated strings
  */ 	
  protected StringSO _StringConcatenate(dSelfObject arg) throws dSelfException{
  
    checkIfStringSO("_StringConcatenate:", arg); 
    return new StringSO(string.concat(((StringSO)arg).string));
  }

 /**
  * Return the substring beginning from the specified position to
  * the end of this string. If the position is to low or high, a 
  * badIndexError will return.
  *
  * @param from The position where the substring begins. The argument 
  * type must be IntegerSO and is checked by the method
  */
  protected StringSO _StringGetSubFrom(dSelfObject from) throws dSelfException{

    checkIfIntegerSO("_StringGetSubFrom:", from); 

    try{
      return new StringSO(string.substring(((IntegerSO)from).value));
    }
    catch(IndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive \"_StringSubFrom:\""+
          " is out of bounds !", "_StringGetSubFrom:", "badIndexError"); 
    }  
  }

 /**
  * Return the substring beginning from the specified position to
  * the specified second position of this string. If the positions are 
  * to low or high, a badIndexError will return.
  *
  * @param from The position where the substring begins. The argument 
  * type must be IntegerSO and is checked by the method
  * @param to The position where the substring ends. The argument 
  * type must be IntegerSO and is checked by the method
  */
  protected StringSO _StringGetSubFromTo(dSelfObject from, dSelfObject to) throws dSelfException{

    checkIfIntegerSO("_StringGetSubFrom:To: (1st argument)", from); 
    checkIfIntegerSO("_StringGetSubFrom:To: (2nd argument)", to); 

    try{
      return new StringSO(string.substring(((IntegerSO)from).value, 
                                         ((IntegerSO)to).value));
    }
    catch(IndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive \"_StringSubFrom:\""+
                " is out of bounds !", "_StringGetSubFrom:To:", "badIndexError"); 
    }  
  }

 /**
  * Prints this string at the front-end, that was chosen by the user.
  *
  * @return This string
  */
  protected StringSO _StringPrint(){
  
    dSelfVM.printMessage(string);
    return this;
  }

 /**
  * Returns the size (number of characters) of this string.
  */
  protected IntegerSO _StringSize(){
  
    return new IntegerSO(string.length());
  }
}
