package parser;

import java_cup.runtime.*;
import dSelf.PrimMsg;
import dSelf.Globals;
import dSelfComp;
import dSelfVM;

%%
 
%class dSelfScanner
%public
%unicode
%line  
%column 
%cup

%{
    
 /**
  * symbol nimmt einen Symbolwert an (siehe sym.java) und gibt
  * diesen als ein Symbolobjekt mit Informationen zur Spalten-
  * und Zeilenposition im Quelltext, sowie einem Objekt, da
  * z.B. den Wert eines Integers oder den Namen einer Nachricht
  * enthlt zurck. Wenn der Scanner sich im Debugmodus befindet, 
  * werden zustzlich die Tokendaten ausgegeben.
  */

 /**
  * Creats a symbol of the specified type with the given value.
  * The current informations about the line and column are also
  * added. When {@link dSelf.Globals#debug_scanner} is activated,
  * then the tokens are printed. 
  *
  * @param type The type of the token symbol
  * @param value The value of the token symbol
  * @return A symbol object that unites the given informations
  */
  protected Symbol symbol(int type, Object value){
   
    if(Globals.debug_scanner){  
    
      System.out.print("line: "+(yyline+1)+",\tcolumn: "+(yycolumn+1)+",\ttoken:");
      switch(type){
    
        case sym.TRUE: printMessage(" true"); break;
        case sym.FALSE: printMessage(" false"); break;
        case sym.NIL: printMessage(" nil"); break;
        case sym.CIRCUMFLEX: printMessage(" ^"); break;
        case sym.IDENTIFIER: printMessage(" identifier(\""+value+"\")"); break;
        case sym.SMALLKEYWORD: printMessage(" small keyword(\""+value+"\")"); break;
        case sym.CAPKEYWORD: printMessage(" capital keyword(\""+value+"\")"); break;
        case sym.ARGUMENTNAME: printMessage(" argument name(\""+value+"\")"); break;
        case sym.STRING: printMessage(" string(\""+value+"\")"); break;
        case sym.OPERATOR: printMessage(" operator(\""+value+"\")"); break;
        case sym.INTEGER: printMessage(" integer("+value+")"); break;
        case sym.LONG: printMessage(" long("+value+")"); break;
        case sym.SHORT: printMessage(" short("+value+")"); break;
        case sym.FLOAT: printMessage(" float("+value+")"); break;
        case sym.DOUBLE: printMessage(" double("+value+")"); break;
        case sym.LEFTPAR: printMessage(" ("); break;
        case sym.RIGHTPAR: printMessage(" )"); break;
        case sym.SELF_KEYWORD: printMessage(" self"); break;
        case sym.DOT: printMessage(" ."); break;
        case sym.EQUAL: printMessage(" ="); break;
        case sym.LEFTARROW: printMessage(" <-"); break;
        case sym.LEFTBRACE: printMessage(" {"); break;
        case sym.RIGHTBRACE: printMessage(" }"); break;
        case sym.LEFTSQUAREBRACKET: printMessage(" ["); break;
        case sym.RIGHTSQUAREBRACKET: printMessage(" ]"); break;
        case sym.VERTICALBAR: printMessage(" |"); break;
        case sym.ASTERISK: printMessage(" *"); break;
        case sym.EMPTY_OBJECT: printMessage(" empty object"); break;
        case sym.UNARYINDIRECTEDRESEND: printMessage(" unary indirected resend(\""+value+"\")"); break;
        case sym.UNARYDIRECTEDRESEND: printMessage(" unary directed resend(\""+value+"\")"); break;
        case sym.BINARYINDIRECTEDRESEND: printMessage(" binary indirected resend(\""+value+"\")"); break;
        case sym.BINARYDIRECTEDRESEND: printMessage(" binary directed resend(\""+value+"\")"); break;
        case sym.KEYWORDINDIRECTEDRESEND: printMessage(" keyword indirected resend(\""+value+"\")"); break;
        case sym.KEYWORDDIRECTEDRESEND: printMessage(" keyword directed resend(\""+value+"\")"); break;
      }	
    }

    return new Symbol(type, yyline, yycolumn, value);
  } 

    protected void printMessage(String msg)
    {
	if (!Globals.genJava) // called from VM
	    dSelfVM.printMessage(msg);
	else
	    System.out.println(msg);
    }  
 /**
  * calls {@link #symbol(int, Object)} with null as value auf. 
  *
  * @param type The type of the token symbol
  * @return A symbol object that unites the given informations
  */
  protected Symbol symbol(int type){

    return symbol(type, null);
  }

/**
 * Calculates for a given string in the form 'xry' or 'xRy' the 
 * corresponding integer value, where x is a number between 2
 * and 36 (the radix) and y a sequence of digits or letters. E.g.
 * 2r11111, 8r37, 10r31, 16r1f or 36rv all represent the number 31.
 *
 * @param str A string in radix-form
 * @return The corresponding integer value.
 */
 protected int radInt2decInt(String str){
  
    boolean isPositive = true;
    
    // Merken, ob Zahl positiv oder negativ ist  
    if(str.charAt(0) == '-'){
      isPositive = false;  
      str = str.substring(1,str.length());
    }
    
    // r ist die Position des 'r' oder 'R' im String  
    int r = str.indexOf('r') == -1 ? str.indexOf('R') : str.indexOf('r');
		
    // Den String in eine Zahl umwandeln	      
    return string2int((isPositive ? "" : "-") + str, 
        (isPositive ? "" : "-") + str.substring(r+1), 
        Integer.parseInt(str.substring(0, r))); 
  } 

/**
 * Calculates for a given string in radix form the corresponding 
 * long value. The procedure is the same as described at 
 * {@link #radInt2decInt(String)}.
 *
 * @param str A string in radix-form
 * @return The corresponding integer value.
 */
 protected long radLong2decLong(String str){
  
    boolean isPositive = true;
    
    // Merken, ob Zahl positiv oder negativ ist  
    if(str.charAt(0) == '-'){
      isPositive = false;  
      str = str.substring(1,str.length());
    }
    
    // r ist die Position des 'r' oder 'R' im String  
    int r = str.indexOf('r') == -1 ? str.indexOf('R') : str.indexOf('r');
		
    // Den String in eine Zahl umwandeln	      
    return string2long((isPositive ? "" : "-") + str, 
        (isPositive ? "" : "-") + str.substring(r+1), 
        Integer.parseInt(str.substring(0, r))); 
  } 

/**
 * Calculates for a given string in radix form the corresponding 
 * short value. The procedure is the same as described at 
 * {@link #radInt2decInt(String)}.
 *
 * @param str A string in radix-form
 * @return The corresponding integer value.
 */
 protected short radShort2decShort(String str){
  
    boolean isPositive = true;
    
    // Merken, ob Zahl positiv oder negativ ist  
    if(str.charAt(0) == '-'){
      isPositive = false;  
      str = str.substring(1,str.length());
    }
    
    // r ist die Position des 'r' oder 'R' im String  
    int r = str.indexOf('r') == -1 ? str.indexOf('R') : str.indexOf('r');
		
    // Den String in eine Zahl umwandeln	      
    return string2short((isPositive ? "" : "-") + str, 
        (isPositive ? "" : "-") + str.substring(r+1), 
        Integer.parseInt(str.substring(0, r))); 
  } 

 /**
  * Converts a string that consists of digits and letters in a
  * decimal integer number. The radix is a number between 2 and 36
  * (e.g. 16 is hexadecimal system) and numstr is the string, that
  * should be converted. 
  *
  * @param radix A number between 2 and 36
  * @param numStr The string, that should be converted
  */  
  protected int string2int(String numStr, int radix){
  
    return string2int(numStr, numStr, radix);
  }
 
 /**
  * Converts a string that consists of digits and letters in a
  * decimal long number. The procedure is the same as described
  * at {@link #string2int(String, int)}.
  *
  * @param radix A number between 2 and 36
  * @param numStr The string, that should be converted
  */  
  protected long string2long(String numStr, int radix){
  
    return string2long(numStr, numStr, radix);
  }

 /**
  * Converts a string that consists of digits and letters in a
  * decimal short number. The procedure is the same as described
  * at {@link #string2int(String, int)}.
  *
  * @param radix A number between 2 and 36
  * @param numStr The string, that should be converted
  */  
  protected short string2short(String numStr, int radix){
  
    return string2short(numStr, numStr, radix);
  }

 /**
  * This method does the same like {@link #string2int(String, int)},
  * but in case of an error an more detailed error message is printed
  * with a reference of its occurence.
  *
  * @param origStr The string, where numStr was embedded
  * @param numStr The string, that should be converted
  * @param radix A number between 2 and 36
  */
  protected int string2int(String origStr, String numStr, int radix){

    // Die Basis mu zwischen 2 und 36 liegen
    if(radix < 2 || radix > 36){ 
      dSelfComp.reportError("The radix of "+origStr+" isn't between 2 and 36 !",
                          yyline, yycolumn); 
      
      // When an error occured, then guess the radix as the maximium
      // possible.
      radix=36;
    }
    
    int result=0;
    
    // Try to convert the string to a number by using integers parseInt
    // method. If an error occures in form of an exception, then convert
    // it again "manually" in order to find the exact position, where
    // the error occured and give more detailed informations back.
    try{  
      result = Integer.parseInt(numStr, radix);
    }catch(NumberFormatException e){

      int endPos = numStr.length()-1;  
      int actPos = endPos;
      int digit = 0;
      result =0;
      
      while(actPos >= 0){
        if((digit = Character.digit(numStr.charAt(actPos), radix)) == -1
	    && numStr.charAt(actPos) != '-')
          dSelfComp.reportError("The digit "+numStr.charAt(actPos)+" in "+origStr
	                       +" is to big !", yyline, yycolumn);             
        else 
          result += Math.pow(radix, endPos-actPos)*digit;
  
        actPos--;       
      }
    }

    return result;
    
  }

 /**
  * This method does the same like {@link #string2long(String, int)},
  * but in case of an error an more detailed error message is printed
  * with a reference of its occurence.
  *
  * @param origStr The string, where numStr was embedded
  * @param numStr The string, that should be converted
  * @param radix A number between 2 and 36
  */
  protected long string2long(String origStr, String numStr, int radix){

    // Die Basis mu zwischen 2 und 36 liegen
    if(radix < 2 || radix > 36){ 
      dSelfComp.reportError("The radix of "+origStr+" isn't between 2 and 36 !",
                          yyline, yycolumn); 
      
      // When an error occured, then guess the radix as the maximium
      // possible.
      radix=36;
    }
    
    long result=0;
    
    // Try to convert the string to a number by using integers parseInt
    // method. If an error occures in form of an exception, then convert
    // it again "manually" in order to find the exact position, where
    // the error occured and give more detailed informations back.
    try{  
      result = Long.parseLong(numStr, radix);
    }catch(NumberFormatException e){

      int endPos = numStr.length()-1;  
      int actPos = endPos;
      int digit = 0;
      result =0;
      
      while(actPos >= 0){
        if((digit = Character.digit(numStr.charAt(actPos), radix)) == -1
	    && numStr.charAt(actPos) != '-')
          dSelfComp.reportError("The digit "+numStr.charAt(actPos)+" in "+origStr
	                       +" is to big !", yyline, yycolumn);             
        else 
          result += Math.pow(radix, endPos-actPos)*digit;
  
        actPos--;       
      }
    }
     
    return result;
  }
  
 /**
  * This method does the same like {@link #string2short(String, int)},
  * but in case of an error an more detailed error message is printed
  * with a reference of its occurence.
  *
  * @param origStr The string, where numStr was embedded
  * @param numStr The string, that should be converted
  * @param radix A number between 2 and 36
  */
  protected short string2short(String origStr, String numStr, int radix){

    // Die Basis mu zwischen 2 und 36 liegen
    if(radix < 2 || radix > 36){ 
      dSelfComp.reportError("The radix of "+origStr+" isn't between 2 and 36 !",
                          yyline, yycolumn); 
      
      // When an error occured, then guess the radix as the maximium
      // possible.
      radix=36;
    }
    
    short result=0;
    
    // Try to convert the string to a number by using integers parseInt
    // method. If an error occures in form of an exception, then convert
    // it again "manually" in order to find the exact position, where
    // the error occured and give more detailed informations back.
    try{  
      result = Short.parseShort(numStr, radix);
    }catch(NumberFormatException e){

      int endPos = numStr.length()-1;  
      int actPos = endPos;
      int digit = 0;
      result =0;
      
      while(actPos >= 0){
        if((digit = Character.digit(numStr.charAt(actPos), radix)) == -1
	    && numStr.charAt(actPos) != '-')
          dSelfComp.reportError("The digit "+numStr.charAt(actPos)+" in "+origStr
	                       +" is to big !", yyline, yycolumn);             
        else 
          result += Math.pow(radix, endPos-actPos)*digit;
  
        actPos--;       
      }
    }
     
    return result;
  }  

  /** A dSelf string, that will be convertet to a Java string.*/
  protected StringBuffer string = new StringBuffer();
    
%}

WhiteSpace       = \r | \n | \r\n | \t | " "
SmallLetter      = [a-z]
CapLetter        = [A-Z]
Digit            = [0-9]
Letter           = {SmallLetter} | {CapLetter}
GeneralDigit     = {Digit} | {Letter}
Base             = {Decimal} ("r" | "R")
Decimal          = {Digit}+
RadixInteger     = "-"? {Base} {GeneralDigit}+ 
Integer          = "-"? {Decimal}
RadixLong        = "-"? {Base} {GeneralDigit}+ ("_l" | "_L")
Long             = {Integer} ("_l" | "_L")
RadixShort       = "-"? {Base} {GeneralDigit}+ ("_s" | "_S")
Short            = {Integer} ("_s" | "_S")
Float            = "-"? (({Decimal} "." {Decimal}) | ({Decimal} ("." {Decimal})? ("e" | "E") ("+" | "-")? {Decimal}))
Double           = {Float} ("_d" | "_D")
Identifier       = ({SmallLetter} | "_")  ({Letter} | {Digit} | "_")*
SmallKeyword     = {Identifier} ":"
CapKeyword       = {CapLetter} ({Letter} | {Digit} | "_")* ":"
ArgumentName     = ":" {Identifier}
OpChar           = "!" | "@" | "#" | "$" | "%" | "&" | "*" | "-" | "+" | "=" | "~" | "/" | "?" | "<" | ">" | "," | ";" | "\\" | "^" | "|"
Operator         = {OpChar}+
Comment          = \" {CommentChar}* \"
CommentChar      = [^\"]               
EmptySlots       = ("|" {WhiteSpace}* "|") | "||"  
EmptyObject      = ("(" {WhiteSpace}* ")") | ("(" {EmptySlots} ")") 
UnaryIndirectedResend   = "resend." {Identifier} 
UnaryDirectedResend     = {Identifier} "." {Identifier} 
BinaryIndirectedResend  = "resend." {OpChar} 
BinaryDirectedResend    = {Identifier} "." {OpChar} 
KeywordIndirectedResend = "resend." {SmallKeyword} 
KeywordDirectedResend   = {Identifier} "." {SmallKeyword} 

%state STRING

%%
 
<YYINITIAL> {
 
"'"                { string.setLength(0); yybegin(STRING); }
"true"             { return symbol(sym.TRUE); }  
"false"            { return symbol(sym.FALSE); }  
"nil"              { return symbol(sym.NIL); }  
"("                { return symbol(sym.LEFTPAR); } 
")"                { return symbol(sym.RIGHTPAR); } 
"["                { return symbol(sym.LEFTSQUAREBRACKET); } 
"]"                { return symbol(sym.RIGHTSQUAREBRACKET); } 
"{"                { return symbol(sym.LEFTBRACE); } 
"}"                { return symbol(sym.RIGHTBRACE); } 
"."                { return symbol(sym.DOT); }   
"="                { return symbol(sym.EQUAL); } 
"^"                { return symbol(sym.CIRCUMFLEX); }  
"<-"               { return symbol(sym.LEFTARROW); }   
"self"             { return symbol(sym.SELF_KEYWORD); }  
"|"                { return symbol(sym.VERTICALBAR); } 
"*"                { return symbol(sym.ASTERISK); } 
{Operator}         { return symbol(sym.OPERATOR, yytext()); }
{Identifier}       { return symbol(sym.IDENTIFIER, yytext()); }
{SmallKeyword}     { return symbol(sym.SMALLKEYWORD, yytext()); }
{CapKeyword}       { return symbol(sym.CAPKEYWORD, yytext()); }
{ArgumentName}     { return symbol(sym.ARGUMENTNAME, yytext().substring(1, yytext().length())); }
{EmptyObject}      { return symbol(sym.EMPTY_OBJECT); }

{UnaryIndirectedResend}   { if(yytext().charAt(7) == '_')
                              dSelfComp.reportError("Resending of primitives is not "+
                                  "allowed !", yyline, yycolumn); 
                             return symbol(sym.UNARYINDIRECTEDRESEND, yytext().substring(7)); 
			  }

{UnaryDirectedResend}     { if(yytext().charAt(yytext().indexOf('.')+1) == '_')
                              dSelfComp.reportError("Resending of primitives is not "+
                                  "allowed !", yyline, yycolumn); 
                             return symbol(sym.UNARYDIRECTEDRESEND, yytext()); 
			  }
			  
{BinaryIndirectedResend}  { return symbol(sym.BINARYINDIRECTEDRESEND, yytext().substring(7)); }
{BinaryDirectedResend}    { return symbol(sym.BINARYDIRECTEDRESEND, yytext()); }

{KeywordIndirectedResend} { if(yytext().charAt(7) == '_')
                              dSelfComp.reportError("Resending of primitives is not "+
                                  "allowed !", yyline, yycolumn); 
                             return symbol(sym.KEYWORDINDIRECTEDRESEND, yytext().substring(7)); 
			  }

{KeywordDirectedResend}   { if(yytext().charAt(yytext().indexOf('.')+1) == '_')
                              dSelfComp.reportError("Resending of primitives is not "+
                                  "allowed !", yyline, yycolumn); 
			     return symbol(sym.KEYWORDDIRECTEDRESEND, yytext()); 
			  }

{RadixInteger}     { return symbol(sym.INTEGER, new Integer(radInt2decInt(yytext()))); } 

{Integer}          { int i=0;
                     try{  
                       i = Integer.parseInt(yytext());
                     }catch(NumberFormatException e){
                       i = string2int(yytext(), 10); 
                     }
                     return symbol(sym.INTEGER, new Integer(i)); 
		   } 

{RadixLong}        { return symbol(sym.LONG, new Long(radLong2decLong(yytext().substring(0, yytext().length()-2)))); } 
{Long}             { long l=0;
                     try{  
                       l = Long.parseLong(yytext().substring(0, yytext().length()-2));
                     }catch(NumberFormatException e){
                       l = string2long(yytext().substring(0, yytext().length()-2), 10); 
                     }
                     return symbol(sym.LONG, new Long(l)); 
		   } 

{RadixShort}       { return symbol(sym.SHORT, new Short(radShort2decShort(yytext().substring(0, yytext().length()-2)))); } 
{Short}            { short s=0;
                     try{  
                       s = Short.parseShort(yytext().substring(0, yytext().length()-2));
                     }catch(NumberFormatException e){
                       s = string2short(yytext().substring(0, yytext().length()-2), 10); 
                     }
                     return symbol(sym.SHORT, new Short(s)); 
		   } 
		   
{Float}            { // Well formed Self literals are syntacticaly a subset of the
                     // well formed Java literals.
                     return symbol(sym.FLOAT, Float.valueOf(yytext()+'f')); 
		   }
		    
{Double}           { return symbol(sym.DOUBLE, Double.valueOf(yytext().substring(0, yytext().length()-2)+'d')); }

{Comment}          { /* skip */ }  
{WhiteSpace}       { /* skip */ } 

.                  { dSelfComp.reportError("Character \""+yytext()+"\" is not "+
                         "allowed there !", yyline, yycolumn); 
		   }
				      
}
   
// Strings are handled seperatly
<STRING> { 

"'"     { yybegin(YYINITIAL); return symbol(sym.STRING, string.toString());}
\\t     { string.append('\t'); }    // Tabulator
\\v     { string.append('\u000b'); } // Vertical tabulator 
\\b     { string.append('\b'); }    // Backspace
\\n     { string.append('\n'); }    // Newline
\\f     { string.append('\f'); }    // Formfeed
\\r     { string.append('\r'); }    // Carriage return
\\0     { string.append('\0'); }    // Null-character
\\\\    { string.append('\\'); }    // Backslash
\\'     { string.append('\''); }    // Single quote
\\\"    { string.append('\"'); }    // Quotation marks
"\?"    { string.append('?'); }     // Question mark   

// '\a' in Self (oder C) don't exists in Java

// Decimal escape-character
\\d[0-9][0-9][0-9][0-9][0-9]                    
        { int iVal = (new Integer(yytext().substring(2))).intValue();
          if(iVal > 65535)
             dSelfComp.reportError("Illegal escape-character \""+yytext()+"\" in"+
	         "String. \n-- Allowed maximum is: \\d65535", yyline, yycolumn);
	  string.append((char)iVal);		

        }

// Hexadecimal escape-character
\\x[0-9a-fA-F][0-9a-fA-F][0-9a-fA-F][0-9a-fA-F] 
        { string.append((char)string2int(yytext().substring(2),
	      yytext().substring(2),16));
	}
		
// Oktal escape-character
\\o[0-8][0-8][0-8][0-8][0-8][0-8]               
        { int iVal=string2int(yytext().substring(2),yytext().substring(2),8);
          if(iVal > 65535){
            dSelfComp.reportError("Illegal escape-character \""+yytext()+"\" in"+
	        "String. \n-- Allowed maximum is: \\o177777", yyline, yycolumn);
	  }	
	  string.append((char)iVal);		
	 }    

// Check if the size of the escape-character is correct
\\d[0-9][0-9][0-9][0-9] |
\\d[0-9][0-9][0-9] | 
\\d[0-9][0-9] |
\\d[0-9] |
\\x[0-9a-fA-F][0-9a-fA-F][0-9a-fA-F] |
\\x[0-9a-fA-F][0-9a-fA-F] |
\\x[0-9a-fA-F] |
\\o[0-8][0-8][0-8][0-8][0-8] |
\\o[0-8][0-8][0-8][0-8] |
\\o[0-8][0-8][0-8] |
\\o[0-8][0-8] |
\\o[0-8]
        { dSelfComp.reportError("Illegal escape-character \""+yytext()+"\" in"+
	      " String. The Number must begin with 0's to have"+
	      " the right size, e.g. \\x00ff instead of \\xff." );
	}

// Except the prefixes \d \x \o and the above listet escape-characters are
// no others allowed
\\[^dxo]   { dSelfComp.reportError("Illegal escape-character \"" + yytext() +
                 "\" in String", yyline, yycolumn);
             }

// Every else sign will be appended to the string
[^\'\r\n]    { string.append(yytext()); }

}
